use std::fmt;

use crate::{error::*, get_release_txn, Lease};
use etcd_client::Client;
use log::*;

pub struct LockGuard<'l, 'n> {
    pub(crate) lock_value: [u8; 16],
    pub(crate) name: &'n str,
    pub(crate) lease: &'l Lease,
}

impl<'l, 'n> fmt::Debug for LockGuard<'l, 'n> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.debug_struct("LockGuard")
            .field("name", &self.name)
            .field("value", &self.lock_value)
            .field("lease_id", &self.lease.lease_id)
            .finish()
    }
}

impl<'n, 'l> LockGuard<'n, 'l> {
    pub async fn release(self, client: &mut Client) -> CrateResult<()> {
        let Self {
            lock_value,
            name,
            lease,
        } = self;

        let release_response = client
            .txn(get_release_txn(name, &lock_value, lease.lease_id))
            .await?;

        if !release_response.succeeded() {
            return Err(Error::Lost);
        } else {
            debug!(
                "Released lock: {} (lease: {:#x})",
                name, self.lease.lease_id
            );
        }

        Ok(())
    }
}
