#![deny(missing_docs)]

/*!
Some utilities for parsing some format based on nested lists into arbitrary data structures.
It's also meant to be used as a backend for parsers.
**/

pub use token_parser_derive::*;

#[derive(Debug)]
/// The error type for token parsing.
pub enum Error {
    /// The sublist contains less elements than expected by a specified amount.
    NotEnoughElements(usize),
    /// The sublist contains more elements than expected by a specified amount.
    TooManyElements(usize),
    /// No list is allowed in this context.
    ListNotAllowed,
    /// No symbol is allowed in this context.
    SymbolNotAllowed,
    /// Error with string parsing.
    StringParsing,
    /// Some specific element is invalid.
    InvalidElement,
}

/// The result type for token parsing.
pub type Result<T> = std::result::Result<T, Error>;

/// Some unit, which represents an intermediate state.
pub enum Unit {
    /// The current unit is a single symbol.
    Symbol(String),
    /// The current unit is a parser, which can yield multiple units.
    Parser(Parser),
}

impl Unit {
    /// Returns the symbol, if applicable, as a result type.
    pub fn symbol(self) -> Result<String> {
        use Unit::*;
        match self {
            Symbol(name) => Ok(name),
            Parser(_) => Err(Error::ListNotAllowed),
        }
    }

    /// Returns the parser, if applicable, as a result type.
    pub fn parser(self) -> Result<Parser> {
        use Unit::*;
        match self {
            Symbol(_) => Err(Error::SymbolNotAllowed),
            Parser(parser) => Ok(parser),
        }
    }
}

/// This trait needs to be implemented for every struct which can be parsed using the token parser.
pub trait Parsable<C>: Sized {
    /// When a symbol is found by the parser, this will be called.
    fn parse_symbol(_name: String, _context: &C) -> Result<Self> {
        Err(Error::SymbolNotAllowed)
    }

    /// When a subparser is found by the parser, this will be called.
    fn parse_list(_parser: &mut Parser, _context: &C) -> Result<Self> {
        Err(Error::ListNotAllowed)
    }
}

fn parse<C, P: Parsable<C>>(unit: Unit, context: &C) -> Result<P> {
    use Unit::*;
    match unit {
        Symbol(name) => Parsable::parse_symbol(name, context),
        Parser(mut parser) => parser.parse_rest(context),
    }
}

impl<C, T: Parsable<C>> Parsable<C> for Box<T> {
    fn parse_symbol(name: String, context: &C) -> Result<Self> {
        Ok(Box::new(Parsable::parse_symbol(name, context)?))
    }

    fn parse_list(parser: &mut Parser, context: &C) -> Result<Self> {
        Ok(Box::new(parser.parse_list(context)?))
    }
}

impl<C, T: Parsable<C>> Parsable<C> for Vec<T> {
    fn parse_list(parser: &mut Parser, context: &C) -> Result<Self> {
        let Parser { form, count } = parser;
        let result = form
            .drain(..)
            .rev()
            .map(|unit| {
                *count += 1;
                parse(unit, context)
            })
            .collect();
        result
    }
}

impl<C> Parsable<C> for String {
    fn parse_symbol(name: String, _context: &C) -> Result<Self> {
        Ok(name)
    }
}

#[macro_export]
/// Derives `Parsable` from symbol for types which implement `FromStr`.
macro_rules! derive_symbol_parsable {
    ($t:ty) => {
        impl<C> Parsable<C> for $t {
            fn parse_symbol(name: String, _context: &C) -> Result<Self> {
                if let Some(value) = name.parse().ok() {
                    Ok(value)
                } else {
                    Err(Error::StringParsing)
                }
            }
        }
    };
    ($t:ty, $($rest:ty),+) => {
        derive_symbol_parsable!($t);
        derive_symbol_parsable!($($rest),+);
    };
}

derive_symbol_parsable!(i8, i16, i32, i64, i128);
derive_symbol_parsable!(u8, u16, u32, u64, u128);
derive_symbol_parsable!(f32, f64);
derive_symbol_parsable!(usize);
derive_symbol_parsable!(bool);

/// The token parser to parse the units into wanted types.
pub struct Parser {
    form: Vec<Unit>,
    count: usize,
}

impl Parser {
    /// Creates a new parser from a list of objects.
    pub fn new<I: IntoIterator>(form: I) -> Self
    where
        I::Item: Into<Unit>,
    {
        let mut form: Vec<_> = form.into_iter().map(|unit| unit.into()).collect();
        form.reverse();
        Self { form, count: 0 }
    }

    /// Tries to parse the next unit as the required type.
    pub fn parse_next<C, T: Parsable<C>>(&mut self, context: &C) -> Result<T> {
        self.count += 1;
        if let Some(token) = self.form.pop() {
            parse(token.into(), context)
        } else {
            Result::Err(Error::NotEnoughElements(self.count))
        }
    }

    /// Tries to parse the rest of the current list into the required type.
    /// If not every available token is used, this will be an error.
    pub fn parse_rest<C, T: Parsable<C>>(&mut self, context: &C) -> Result<T> {
        let result = self.parse_list(context);
        if let Some(_) = self.form.pop() {
            let mut count = 1;
            while let Some(_) = self.form.pop() {
                count += 1;
            }
            Err(Error::TooManyElements(count))
        } else {
            result
        }
    }

    /// Tries to parse as many tokens of the current list as needed into the required type.
    pub fn parse_list<C, T: Parsable<C>>(&mut self, context: &C) -> Result<T> {
        Parsable::parse_list(self, context)
    }
}

impl Iterator for Parser {
    type Item = Result<Parser>;

    fn next(&mut self) -> Option<Result<Parser>> {
        self.count += 1;
        Some(self.form.pop()?.parser())
    }
}
