#![deny(missing_docs)]
/*!
This crate contains a minimal form of s-expressions, which will be called token lists.

Every element, called token, is either a symbol or a list of multiple tokens.

Other libraries like the token parser can be used to parse tokens.
**/

/// Represents the token lists.
#[derive(Clone, Debug, PartialEq)]
pub enum Token {
    /// A symbol token represented by its name.
    Symbol(String),
    /// A list token containing other tokens.
    List(Vec<Token>),
}

use Token::*;

impl Token {
    /// If the token is a `Symbol`, returns an `Option`, containing the symbol name.
    /// Else returns `None`.
    pub fn symbol(self) -> Option<String> {
        match self {
            Symbol(string) => Some(string),
            List(_) => None,
        }
    }

    /// Same as `fn symbol`, but returns a ref.
    pub fn symbol_ref(&self) -> Option<&String> {
        match self {
            Symbol(string) => Some(string),
            List(_) => None,
        }
    }

    /// If the token is a `List`, returns an `Option`, containing the elements.
    /// Else returns `None`.
    pub fn list(self) -> Option<Vec<Token>> {
        match self {
            Symbol(_) => None,
            List(tokens) => Some(tokens),
        }
    }

    /// Same as `fn list`, but returns a ref.
    pub fn list_ref(&self) -> Option<&Vec<Token>> {
        match self {
            Symbol(_) => None,
            List(tokens) => Some(tokens),
        }
    }
}

impl From<String> for Token {
    fn from(arg: String) -> Token {
        Symbol(arg)
    }
}

impl From<&str> for Token {
    fn from(arg: &str) -> Token {
        Symbol(arg.into())
    }
}

impl<T: Copy + Into<Token>> From<&T> for Token {
    fn from(arg: &T) -> Token {
        (*arg).into()
    }
}

impl<T: Into<Token>> From<Vec<T>> for Token {
    fn from(list: Vec<T>) -> Token {
        List(list.into_iter().map(|token| token.into()).collect())
    }
}

mod implement_display {
    use std::fmt::{Display, Formatter, Result};

    use crate::Token;

    impl Display for Token {
        fn fmt(&self, f: &mut Formatter) -> Result {
            match self {
                Token::Symbol(string) => write!(f, "'{}'", string),
                Token::List(vec) => {
                    let mut first = true;
                    write!(f, "(").expect("Unexpected end of file");
                    for tok in vec.iter() {
                        if !first {
                            write!(f, " ").expect("Unexpected end of file");
                        } else {
                            first = false;
                        }
                        let result = write!(f, "{}", tok);
                        if result.is_err() {
                            return result;
                        }
                    }
                    write!(f, ")")
                }
            }
        }
    }
}

#[cfg(feature = "parser")]
mod parser;
