use glob::glob;
use std::collections::HashMap;
use tokay; // This is Tokay v0.4

static PATTERN: &str = "src/**/*.rs";

fn main() {
    let mut res: HashMap<String, String> = HashMap::new();

    // Use Tokay v0.4
    std::env::set_var("TOKAY_DEBUG", "0"); // disable any debug here.
    let mut compiler = tokay::compiler::Compiler::new();

    let program = compiler
        // todo: use Compiler::compile_str() later here...
        .compile(tokay::reader::Reader::new(Box::new(std::io::Cursor::new(
            include_str!("build.tok"),
        ))))
        .expect("Tokay compile error");

    // This is some little, naive attempt to run the build.tok script on all .rs-source files
    // and obtain the module path from the src/-file- and folder-structure. It surely can be done
    // better, but suffices the current requirements.
    for entry in glob(PATTERN).expect("Failed to read glob pattern") {
        match entry {
            Ok(path) => {
                match program.run_from_string(
                    std::fs::read_to_string(&path)
                        .expect(&format!("Unable to read {}", path.display())),
                ) {
                    Ok(None) => {}
                    Ok(Some(matches)) => {
                        println!("cargo:rerun-if-changed={}", path.display());
                        //let path = path.into_iter().map(|part| part.to_str().unwrap().to_string()).collect::<Vec<String>>();

                        // Generate result entries from all matches of build.tok
                        let matches = matches.to_list();

                        for func in matches {
                            let func = func.borrow().to_dict();
                            let kind = func["kind"].borrow().to_string();
                            let name = func["name"].borrow().to_string();

                            // Generate module prefix from path...
                            let module = path
                                .iter()
                                .enumerate()
                                .map(|(i, part)| {
                                    let mut part = part.to_str().unwrap();

                                    if i == 0 {
                                        return "crate".to_string();
                                    } else if part.ends_with(".rs")
                                    // fixme: can the be done better?
                                    {
                                        // cut away the ".rs" here...
                                        part = &part[..part.len() - 3];

                                        // create "type::Type" here in case it's a method
                                        if kind == "method" {
                                            return format!(
                                                "{}::{}{}",
                                                part,
                                                &part[0..1].to_uppercase(),
                                                &part[1..]
                                            );
                                        }
                                    }

                                    part.to_string()
                                })
                                .collect::<Vec<String>>()
                                .join("::");

                            // Generate full qualified function name
                            res.insert(
                                name.clone(),
                                format!("{}::tokay_{}_{}", module, kind, name.to_lowercase()),
                            );
                        }
                    }
                    Err(_) => panic!("Error during execution"),
                }
            }
            Err(e) => println!("{:?}", e),
        }
    }

    // Sort keys
    let mut keys: Vec<String> = res.keys().map(|key| key.clone()).collect();
    keys.sort();

    // Generate source
    let f = "src/_builtins.rs";
    let s = r#"/*! Tokay builtin registry

THIS MODULE IS AUTOMATICALLY GENERATED BY BUILD.RS;
DON'T CHANGE THIS FILE MANUALLY, IT WILL GO AWAY!!!
*/
use crate::builtin::Builtin;

pub static BUILTINS: [Builtin; ##count] = [
##defs];
"#
    .replace(
        "##defs",
        &keys
            .into_iter()
            .map(|key| {
                format!(
                    "    Builtin {{\n        name: \"{}\",\n        func: {},\n    }},\n",
                    key, res[&key]
                )
            })
            .collect::<Vec<String>>()
            .concat(),
    )
    .replace("##count", &res.len().to_string());

    // Exit when file exists and didn't change
    if let Ok(c) = std::fs::read_to_string(f) {
        if c == s {
            std::process::exit(0);
        }
    }

    std::fs::write(f, s).expect(&format!("Unable to write '{}'", f));
}
