use crate::Result;
use crate::{
    client::{Client, ClientType},
    error::ClientError,
    types::ClientOptions,
};

#[derive(Debug, Clone, PartialEq, Eq)]
pub enum Protocol {
    Handshake,
    Nimiq,
    Bitcoin,
    Ethereum,
    EthereumClassic,
    Ravencoin,
    Siacoin,
    Chia,
    Turtl,
    Starcoin,
    //Only constructed as the default. Will fail operations.
    Unknown,
}

impl Protocol {
    pub fn get_client(
        &self,
        client_type: ClientType,
        options: ClientOptions,
    ) -> Result<Box<Client>> {
        match *self {
            Protocol::Bitcoin => Ok(crate::clients::bitcoin::get_client(client_type, options)?),
            Protocol::Nimiq => Ok(crate::clients::nimiq::get_client(client_type, options)?),
            Protocol::Handshake => Ok(crate::clients::handshake::get_client(client_type, options)?),
            Protocol::Ethereum => Ok(crate::clients::ethereum::get_client(client_type, options)?),
            Protocol::EthereumClassic => Ok(crate::clients::ethereum_classic::get_client(
                client_type,
                options,
            )?),
            Protocol::Ravencoin => Ok(crate::clients::ravencoin::get_client(client_type, options)?),
            Protocol::Siacoin => Ok(crate::clients::siacoin::get_client(client_type, options)?),
            Protocol::Chia => Ok(crate::clients::chia::get_client(client_type, options)?),
            Protocol::Turtl => Ok(crate::clients::turtl::get_client(client_type, options)?),
            Protocol::Starcoin => Ok(crate::clients::starcoin::get_client(client_type, options)?),
            Protocol::Unknown => Err(ClientError::UnknownProtocol),
        }
    }

    //@todo cast everything to lowercase first.
    pub fn from_str(s: &str) -> Protocol {
        match s {
            "handshake" | "hns" => Protocol::Handshake,
            "nimiq" | "nim" => Protocol::Nimiq,
            "bitcoin" | "btc" => Protocol::Bitcoin,
            "ethereum" | "eth" => Protocol::Ethereum,
            "ethereum classic" | "ethereum_classic" | "etc" => Protocol::EthereumClassic,
            "ravencoin" | "raven" | "rvn" => Protocol::Ravencoin,
            "siacoin" | "sc" | "sia" => Protocol::Siacoin,
            "chiacoin" | "chia" => Protocol::Chia,
            "turtl" | "turtlcoin" => Protocol::Turtl,
            "starcoin" | "stc" => Protocol::Starcoin,
            _ => Protocol::Unknown,
        }
    }

    //@todo I think this would be nicer if we just have all of the parts of the enum implement
    //Protocol which requires some set of functions -> Similar to client. But for now, this works.
    pub fn hashrate_multiplier(&self) -> u64 {
        match *self {
            //@todo unconfirmed, but I'm relatively confident. for all below up to Nimiq
            Protocol::Bitcoin => 2_u64.pow(32),
            Protocol::Ethereum => 2_u64.pow(32),
            Protocol::EthereumClassic => 2_u64.pow(32),
            Protocol::Ravencoin => 2_u64.pow(32),
            Protocol::Siacoin => 2_u64.pow(32),
            //@todo unclear how this is calculated since I believe it's mostly space?
            Protocol::Chia => 1,
            //@todo neither of these have the correct values... YEt.
            Protocol::Turtl => 1,
            Protocol::Starcoin => 1,

            //Confirmed
            Protocol::Nimiq => 2_u64.pow(16),
            Protocol::Handshake => 2_u64.pow(32),
            Protocol::Unknown => 1,
        }
    }

    pub fn decimals(&self) -> u32 {
        match *self {
            Protocol::Bitcoin => 8,
            //@todo
            _ => 0,
        }
    }

    pub fn decimal_multiplier(&self) -> u64 {
        match *self {
            Protocol::Bitcoin => 100_000_000,
            //@todo
            _ => 0,
        }
    }
}

impl From<String> for Protocol {
    fn from(s: String) -> Self {
        Protocol::from_str(&s)
    }
}

impl From<&str> for Protocol {
    fn from(s: &str) -> Self {
        Protocol::from_str(s)
    }
}
