use crate::client::{Client, ClientType, CryptoClient};
use crate::types::{Amount, Currency, Symbol, WalletBalance};
use crate::types::{ClientOptions, SyncItem, SyncStats};
use crate::Result;
use async_trait::async_trait;
use num_bigint::BigInt;
use rpc_json_client::{ClientBuilder, RpcClient};
use serde_json::json;
use std::sync::Arc;

#[derive(Clone)]
pub struct NimiqRpcClient {
    client: Arc<RpcClient>,
}

pub fn get_client(client_type: ClientType, options: ClientOptions) -> crate::Result<Box<Client>> {
    match client_type {
        //@todo need url to come as well.
        ClientType::RPC => Ok(Box::new(NimiqRpcClient::new(options))),
        _ => Err(crate::error::ClientError::UnsupportedType(client_type)),
    }
}

impl NimiqRpcClient {
    pub fn new(options: ClientOptions) -> Self {
        //@todo probably expose this with a with_builder as well here.
        let client = ClientBuilder::new(&options.url).with_retry().build();
        NimiqRpcClient {
            client: Arc::new(client),
        }
    }

    pub fn new_with_backups(uri: &str, backups: Vec<String>) -> Self {
        let client = ClientBuilder::new(uri)
            .with_retry()
            .with_backups(backups)
            .build();
        NimiqRpcClient {
            client: Arc::new(client),
        }
    }
}

#[async_trait]
impl CryptoClient for NimiqRpcClient {
    async fn sync_stats(&self) -> SyncStats {
        let result: serde_json::Value = match self.client.execute("syncing", &[]).await {
            Ok(r) => r,
            //@todo we should just move this entire error handling outside of here once we have results
            //in this lib.
            Err(_) => {
                // info!("Node appeared to not be ready yet. Sleeping for 60 seconds");
                async_std::task::sleep(std::time::Duration::from_secs(60)).await;

                return SyncStats {
                    current_block: 0,
                    syncing: true,
                    sync_item: SyncItem::Block,
                    estimated_sync_item_remaining: 0.0,
                };
            }
        };

        if let Some(syncing) = result.as_bool() {
            let result: serde_json::Value = self.client.execute("blockNumber", &[]).await.unwrap();

            let block = result.as_u64().unwrap();

            if syncing == false {
                return SyncStats {
                    current_block: block,
                    syncing: false,
                    sync_item: SyncItem::Block,
                    estimated_sync_item_remaining: 0.0,
                };
            }
        }

        //@todo this will absolutely break when the nimiq node is done syncing... So we need to fix
        //that before it's done.
        let current_block = result
            .as_object()
            .unwrap()
            .get("currentBlock")
            .unwrap()
            .as_u64()
            .unwrap();
        // let tip = result
        //     .as_object()
        //     .unwrap()
        //     .get("highestBlock")
        //     .unwrap()
        //     .as_u64()
        //     .unwrap();

        SyncStats {
            current_block,
            syncing: true,
            sync_item: SyncItem::Block,
            //@todo when there is 0 left here, let's mark this as "unknown" to the admin.
            //Because nimiq is hot garbage, we don't have a solid way of estimating sync time
            //without hardcoding in some kind of highestBlock.
            estimated_sync_item_remaining: 0.0,
        }
    }

    async fn wallet_balance(&self, identifier: &str) -> Result<WalletBalance> {
        let res: u64 = self
            .client
            .execute("getBalance", &[json!(identifier)])
            .await?;

        Ok(WalletBalance {
            //We need bigints here actually.
            confirmed_balance: Amount {
                value: BigInt::from(res),
                currency: Currency {
                    symbol: Symbol::NIM,
                    decimals: 5,
                },
            },
            unconfirmed_balance: Amount {
                value: BigInt::from(res),
                currency: Currency {
                    symbol: Symbol::NIM,
                    decimals: 5,
                },
            },
        })
    }

    fn client_type(&self) -> ClientType {
        ClientType::RPC
    }
}
