use crate::client::{ClientType, ToBClient};
use crate::protocol::Protocol;
use crate::types::ClientOptions;
use crate::Result;

//@todo derive default here please.
pub struct ToBClientBuilder {
    pub url: String,
    pub protocol: Protocol,
    pub client_type: ClientType,
    pub username: Option<String>,
    pub password: Option<String>,
}

impl ToBClientBuilder {
    pub fn new(url: &str) -> Self {
        ToBClientBuilder {
            url: url.to_owned(),
            protocol: Protocol::Unknown,
            client_type: ClientType::RPC,
            username: None,
            password: None,
        }
    }

    pub fn with_protocol<T: Into<Protocol>>(mut self, protocol: T) -> Self {
        self.protocol = protocol.into();
        self
    }

    pub fn with_client_type(mut self, client_type: ClientType) -> Self {
        self.client_type = client_type;
        self
    }

    pub fn with_username(mut self, username: &str) -> Self {
        self.username = Some(username.to_owned());
        self
    }

    pub fn with_password(mut self, password: &str) -> Self {
        self.password = Some(password.to_owned());
        self
    }

    pub fn build(self) -> Result<ToBClient> {
        //@todo we should probably have to pass in here "client options" which will include URL,
        //retry, backups, username, password.
        let client = self.protocol.get_client(
            self.client_type,
            ClientOptions {
                url: self.url.to_owned(),
                username: self.username,
                password: self.password,
            },
        )?;

        Ok(ToBClient {
            inner_client: client,
        })
    }
}
