use crate::client::{Client, ClientType, CryptoClient};
use crate::types::{Amount, ClientOptions, Currency, Symbol, SyncItem, SyncStats, WalletBalance};
use crate::Result;
use async_trait::async_trait;
use num_bigint::BigInt;
use rpc_json_client::{ClientBuilder, RpcClient};
use std::sync::Arc;

#[derive(Clone)]
pub struct HandshakeRpcClient {
    client: Arc<RpcClient>,
}

pub fn get_client(client_type: ClientType, options: ClientOptions) -> crate::Result<Box<Client>> {
    match client_type {
        //@todo need url to come as well.
        ClientType::RPC => Ok(Box::new(HandshakeRpcClient::new(options))),
        _ => Err(crate::error::ClientError::UnsupportedType(client_type)),
    }
}

impl HandshakeRpcClient {
    pub fn new(options: ClientOptions) -> Self {
        let client = ClientBuilder::new(&options.url).with_retry().build();

        HandshakeRpcClient {
            client: Arc::new(client),
        }
    }
}

#[async_trait]
impl CryptoClient for HandshakeRpcClient {
    async fn sync_stats(&self) -> SyncStats {
        let result: serde_json::Value = match self.client.execute("getblockchaininfo", &[]).await {
            Ok(r) => r,
            Err(_e) => {
                //@todo completely revamp this section should probably return a error like I say
                //below, but we want to make sure that the monitor understands to re-try.
                // info!("Node appeared to not be ready yet. Sleeping for 60 seconds");
                async_std::task::sleep(std::time::Duration::from_secs(60)).await;
                //We should return an error here... but oh well
                return SyncStats {
                    current_block: 0,
                    syncing: true,
                    sync_item: SyncItem::VerificationProgress,
                    estimated_sync_item_remaining: 0.0,
                };
            }
        };

        let blocks = result.as_object().unwrap().get("blocks").unwrap();
        //Should be match instead.
        let current_block = blocks.as_u64().unwrap();

        //@todo save the as_object up above and then just reuse that everytime instead of
        //re-unwrapping it everytime.
        //Should be match instead.
        let progress = result
            .as_object()
            .unwrap()
            .get("verificationprogress")
            .unwrap()
            .as_f64()
            .unwrap();

        let estimated_sync_item_remaining = 1.0 - progress;

        //Triple 9s
        let (syncing, estimated_sync_item_remaining) = if estimated_sync_item_remaining < 0.001 {
            (false, 0.0)
        } else {
            (true, estimated_sync_item_remaining)
        };

        SyncStats {
            current_block,
            syncing,
            sync_item: SyncItem::VerificationProgress,
            estimated_sync_item_remaining,
        }
    }

    //@todo be able to use get wallet balance here similar to BTC -> Need to test those..
    async fn wallet_balance(&self, _identifier: &str) -> Result<WalletBalance> {
        let res: f64 = self.client.execute("getbalance", &[]).await?;
        //@todo I think this accepts Identifer but not 100% sure.
        let res_2: f64 = self.client.execute("getunconfirmedbalance", &[]).await?;

        //@todo we should also maybe convert everything to BIGINTS first, before doing math on them
        //so that we don't risk any overflows... But for now I think I'm fine with this.
        let confirmed_balance = BigInt::from((res * 1_000_000.0).floor() as u64);
        let unconfirmed_balance = BigInt::from((res_2 * 1_000_000.0).floor() as u64);

        Ok(WalletBalance {
            //We need bigints here actually.
            confirmed_balance: Amount {
                value: confirmed_balance,
                currency: Currency {
                    symbol: Symbol::HNS,
                    decimals: 6,
                },
            },
            unconfirmed_balance: Amount {
                value: unconfirmed_balance,
                currency: Currency {
                    symbol: Symbol::HNS,
                    decimals: 6,
                },
            },
        })
    }

    fn client_type(&self) -> ClientType {
        ClientType::RPC
    }
}
