use tmo_tools::*;
use serde_derive::Deserialize;
use actix_files::Files;
use actix_web::{web, App, HttpServer, HttpResponse};
use std::{
    env, thread, fmt, process,
    sync::RwLock,
    time::Duration
};
use lazy_static::lazy_static;
use rust_utils::{
    config::Config,
    logging::LogLevel
};

mod pages;

#[derive(Deserialize)]
pub struct SettingsForm {
    // general settings
    gw_ip: String,
    password: String,
    new_net: String,

    // gateway settings
    reboot_on_err: Option<String>,
    auto_reboot: Option<String>,
    reboot_hr: u32,
    reboot_min: u32,

    // connected devices settings
    tld: String,
    gw_name: String
}

lazy_static! {
    static ref STATS: RwLock<Option<stats::GatewayStats>> = RwLock::new(None);
    static ref INFO: RwLock<Option<stats::GatewayInfo>> = RwLock::new(None);
}

#[derive(Debug)]
pub struct ValueError;
impl fmt::Display for ValueError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "Can't get value!")
    }
}

impl actix_web::ResponseError for ValueError { }

#[actix_web::main]
async fn main() -> std::io::Result<()> {
    LOG.report_panics(true);
    LOG.line_basic("Starting up T-Mobile Gateway Monitor Web UI...", true);

    // this thread refreshes the stats 
    thread::spawn(|| {
        lazy_static::initialize(&CLIENT);
        loop {
            if let Ok(mut prev_stats) = STATS.try_write() {
                if let Ok(mut prev_info) = INFO.try_write() {
                    let stats = CLIENT.read().unwrap().get_all_stats();
                    let info = CLIENT.read().unwrap().get_info();
                    *prev_stats = match stats {
                        Ok(ref s) => Some(s.clone()),
                        Err(why) => {
                            LOG.line(LogLevel::Error, format!("Error getting gateway stats: {}", why), true);
                            None
                        }
                    };

                    *prev_info = match info {
                        Ok(i) => Some(i.clone()),
                        Err(why) => {
                            LOG.line(LogLevel::Error, format!("Error getting gateway info: {}", why), true);
                            None
                        }
                    };
                }
            }
            thread::sleep(Duration::from_secs(10));
        }
    });

    ctrlc::set_handler(|| {
        LOG.line_basic("Shutting down...", true);
        process::exit(0);
    }).expect("Error setting Ctrl-C handler");

    LOG.line_basic("Startup complete", true);

    if env::set_current_dir("webui/").is_err() {
        env::set_current_dir("/usr/share/tmobile-internet-tools/webui/").unwrap();
    };
    HttpServer::new(|| {
        //let http_client = awc::Client::default();
        App::new()
            .route("/", web::get().to(pages::index))
            .route("/help", web::get().to(pages::help))
            .route("/adv_stats", web::get().to(pages::adv_stats))
            .route("/conn_stats", web::get().to(pages::conn_stats))
            
            .route("/diag", web::get().to(pages::diag))
            .route("/diag/gatewaymon_log", web::get().to(pages::gatewaymon_log))
            .route("/diag/gateway_stats_log", web::get().to(pages::gateway_stats_log))
            
            .route("/config", web::get().to(pages::config))
            .route("/config/save_cfg", web::post().to(save))
            .route("/config/reboot_gateway", web::get().to(reboot_gateway))
            //.service(web::resource("/gateway_interface/{path:.*}").to(gateway_interface))
            .service(Files::new("/", ".").show_files_listing())
            //.app_data(web::Data::new(http_client))
    })
    .bind(("0.0.0.0", 7700))?
    .run()
    .await
}

// request handlers

// reboot the gateway
async fn reboot_gateway() -> HttpResponse {
    let res = client_req(|| CLIENT.write().unwrap().reboot_gateway(true));
    if res.is_err() {
        HttpResponse::from_error(res.unwrap_err())
    }
    else {
        HttpResponse::Ok().finish()
    }
}

// save config
async fn save(req: web::Form<SettingsForm>) -> HttpResponse {
    let form = req.into_inner();
    let client = &mut CLIENT.write().unwrap();
    client.config.gateway_ip = form.gw_ip;
    client.config.password = form.password;
    if !form.new_net.is_empty() {
        client.config.valid_networks.push(form.new_net);
    }
    LOG.line_basic("Saving global settings...", true);
    client.config.save().expect("Unable to save settings!");
    client.gw_config.reboot_on_err = form.reboot_on_err.is_some();
    client.gw_config.auto_reboot = form.auto_reboot.is_some();
    client.gw_config.reboot_hr = form.reboot_hr;
    client.gw_config.reboot_min = form.reboot_min;
    LOG.line_basic("Saving gateway settings...", true);
    client.gw_config.save().expect("Unable to save settings!");
    client.dev_config.tld = form.tld;
    client.dev_config.gateway_name = form.gw_name;
    LOG.line_basic("Saving device settings...", true);
    client.dev_config.save().expect("Unable to save settings!");
    LOG.line_basic("Reloading settings...", true);
    client.reload_config();

    // the "created" resource is the reloaded settings
    HttpResponse::Created().append_header(("Location", "/config"))
        .content_type("text/html")
        .body(
            "<!DOCTYPE html>\n\
            <html>\n\
                <head>\n\
                    <link rel=\"stylesheet\" href=\"/styles.css\">\n\
                    <meta http-equiv=\"refresh\" content=\"0; url='/config'\"/>\n\
                </head>\n\
                <body>\n\
                    <a href=\"/config\">Reload settings</a>\n\
                </body>\n\
            </html>"
        )
}

// reverse proxy to the native gateway web interface
/*async fn gateway_interface(path: web::Path<String>, req: HttpRequest, body: web::Payload, http_client: Data<awc::Client>) -> HttpResponse {
    let gw_ip = &CLIENT.read().unwrap().config.gateway_ip;
    let url = format!("http://{gw_ip}/{path}");
    let mut new_head = req.head().clone();
    new_head.uri = url.parse().unwrap();
    let new_req = http_client
        .request_from(&url, &new_head)
        .insert_header((header::REFERER, ""))
        .insert_header((header::HOST, gw_ip.clone()));
    println!("{}", url);
    println!("{:#?}", new_req);
    match new_req.send().await {
        Ok(resp) => {
            let status = resp.status();

            let mut resp_builder = HttpResponse::build(status);
            for header in resp.headers() {
                resp_builder.insert_header(header);
            }
            resp_builder.streaming(resp.into_stream())
        },
        Err(why) => {
            println!("{}", why);
            HttpResponse::build(StatusCode::BAD_GATEWAY).body("Bad Gateway")
        }
    }
}*/

// the raw gateway request must be made on another thread, otherwise the server crashes
fn client_req<T: Send + Sync + 'static, F: FnOnce() -> T + Send + Sync + 'static>(request: F) -> T {
    let thread = thread::spawn(move || request());
    thread.join().unwrap()
}