use super::*;
use std::collections::HashSet;
use serde_derive::{Serialize, Deserialize};

lazy_static! {
    static ref DIR: String = format!("{}/.local/share/tmo-tools", env::var("HOME").expect("Where the hell is your home folder?!"));
}

// gateway signal tracker
// hopefully this will be able to help
// me find out what the hell is going
// on with this thing's connectivity!
#[derive(Serialize, Deserialize, Clone, Eq, PartialEq)]
pub struct SignalTracker {
    pub entries: Vec<SignalTrackerEntry>
}

impl SignalTracker {
    pub fn get() -> SignalTracker {
        let encoded = match fs::read(format!("{}/signal_tracker", &*DIR)) {
            Ok(f) => f,
            Err(_) => {
                return SignalTracker { entries: vec![] }
            }
        };

        let log = match bincode::deserialize(&encoded[..]) {
            Ok(l) => l,
            Err(why) => {
                LOG.line(LogLevel::Error, why, false);
                return SignalTracker { entries: vec![] }
            }
        };
        log
    }

    pub fn add_entry(&mut self, time: (u32, u32), date: (u32, u32, u32), info_5g: CellInfo, info_4g: CellInfo, conn_stat: bool, reboot: bool) {
        self.entries.push(SignalTrackerEntry {
            time,
            date,
            info_5g,
            info_4g,
            conn_stat,
            reboot
        });
        self.save();
    }

    pub fn get_formatted_entries(&self, date: (u32, u32, u32)) -> (StyledString, StyledString, StyledString) {
        let mut entries = Vec::new();
        let header = "│Time │Connected?│LTE Signal                        │5G Signal                         │Rebooted?│\n";
        let div = "├─────┼──────────┼──────────────────────────────────┼──────────────────────────────────┼─────────┤\n";
        let tail = "└─────┴──────────┴──────────────────────────────────┴──────────────────────────────────┴─────────┘";

        for entry in &self.entries {
            if entry.date != date {
                continue;
            }
            let mut out_str = StyledString::default();
            out_str.append(format!("│{:02}:{:02}│", entry.time.0, entry.time.1));
            out_str.append(if entry.conn_stat { "Yes       │" } else {"No        │"});
            out_str.append(entry.info_4g.fmt_colored());
            out_str.append("│");
            out_str.append(entry.info_5g.fmt_colored());
            out_str.append(if entry.reboot { "│Yes      │\n" } else { "│No       │\n" });
            entries.push(out_str);
            entries.push(StyledString::plain(div));
        }

        entries.pop();
        let mut out = StyledString::new();
        for entry in entries {
            out.append(entry);
        }

        let mut header_out = StyledString::from(header);
        header_out.append(div);

        (header_out, out, StyledString::from(tail))
    }

    pub fn get_dates(&self) -> Vec<(u32, u32, u32)> {
        let mut dates = Vec::new();
        for entry in self.entries.iter().rev() {
            dates.push(entry.date);
        }
        let mut uniques = HashSet::new();
        dates.retain(|d| uniques.insert(*d));
        dates.sort();
        dates.reverse();
        dates
    }

    fn save(&self) {
        let encoded = bincode::serialize(self).unwrap();
        fs::create_dir_all(&*DIR).expect("Unable to save log file!");
        fs::write(format!("{}/signal_tracker", &*DIR), &encoded[..]).expect("Unable to save log file!");
    }
}

// entry in the signal tracker
// hopefully this will be able to help
// me find out what the hell is going
// on with this thing's connectivity!
#[derive(Serialize, Deserialize, Clone, Eq, PartialEq)]
pub struct SignalTrackerEntry {
    // the date the signal info was collected at
    // (year, month, day)
    pub date: (u32, u32, u32),

    // the time the signal info was collected at
    // (hour, minute)
    pub time: (u32, u32),

    // is the gateway claiming that it has connectivity?
    pub conn_stat: bool,

    // cell signal info
    pub info_4g: CellInfo,
    pub info_5g: CellInfo,

    // did the gateway reboot?
    pub reboot: bool
}