use std::{
    fmt, env, fs,
    sync::Arc,
    ffi::OsStr,
    time::Duration,
    process::Command,
    collections::HashSet
};
use reqwest::{
    blocking::Client,
    cookie::Jar
};
use rust_utils::{
    utils,
    config::Config,
    logging::LogLevel
};
use lazy_static::lazy_static;
use token::AuthToken;
use crate::{
    config::{GlobalConfig, GatewayConfig, DeviceConfig},
    devices::DeviceList,
    stats::*,
    LOG,
    DEBUG,
    PRINT_STDOUT
};
use cursive::{
    theme::{BaseColor, Color, ColorStyle},
    utils::markup::StyledString
};

mod token;
pub mod tracker;

lazy_static! {
    static ref LOG_DIR: String = format!("{}/.local/share/tmo-tools", env::var("HOME").expect("Where the hell is your home folder?!"));
}

#[derive(Debug, Clone)]
pub enum ClientError {
    LoginFailed,
    GatewayUnreachable(String),
    JSONParseError(String, String)
}

pub type ClientResult<T> = Result<T, ClientError>;

impl actix_web::ResponseError for ClientError { }

impl fmt::Display for ClientError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match *self {
            Self::LoginFailed => write!(f, "Unable to log into the T-Mobile gateway. Is the username/password in global.toml correct? Or is there a web client logged in elsewhere?"),
            Self::GatewayUnreachable(ref why) => write!(f, "Unable to reach the T-Mobile gateway: {}", why),
            Self::JSONParseError(ref why, ref data) => write!(f, "JSON data failed to parse correctly!\nError: {}\nJSON data:\n{}", why, data)
        }
    }
}

#[derive(Copy, Clone)]
enum AuthMode {
    //App,
    Web,
    None
}

pub struct TrashcanClient {
    pub config: GlobalConfig,
    pub gw_config: GatewayConfig,
    pub dev_config: DeviceConfig,
    client: Client,
    jar: Arc<Jar>,
}

impl TrashcanClient {
    // create a new client
    pub fn new() -> TrashcanClient{
        let config = GlobalConfig::load();
        let jar = Arc::new(Jar::default());
        let client = Client::builder().connect_timeout(Duration::from_secs(10)).timeout(Duration::from_secs(10)).cookie_store(true).cookie_provider(jar.clone()).build().unwrap();
        TrashcanClient {
            config,
            gw_config: GatewayConfig::load(),
            dev_config: DeviceConfig::load(),
            client: client.clone(),
            jar
        }
    }

    // request the gateway to reboot
    pub fn reboot_gateway(&mut self, print_stdout: bool) -> ClientResult<()> {
        LOG.line(LogLevel::Info, "Rebooting gateway...", print_stdout);
        self.reboot_gateway_no_log()
    }

    // request the gateway to reboot (no log output)
    pub fn reboot_gateway_no_log(&mut self) -> Result<(), ClientError> {
        self.post("reboot_web_app.cgi", AuthMode::Web)
    }

    // get a list of devices and their ipv4 addresses connected to the gateway
    pub fn get_devices(&self) -> ClientResult<DeviceList> {
        let response = self.get("dashboard_device_info_status_web_app.cgi", AuthMode::None)?;
        let json = parse_json(&response);
        Ok(DeviceList::get(json?, &self.dev_config)?)
    }

    // get the 5G metrics
    pub fn get_5g_stats(&self) -> ClientResult<CellInfo> {
        let response = self.get("fastmile_radio_status_web_app.cgi", AuthMode::None)?;
        let json = parse_json(&response)?;
        let bars = json["cell_LTE_stats_cfg"][0]["stat"]["RSRPStrengthIndexCurrent"].as_u8().expect("Value not found!");
        let strength = json["cell_LTE_stats_cfg"][0]["stat"]["RSRPCurrent"].as_i32().expect("Value not found!");
        let band = json["cell_5G_stats_cfg"][0]["stat"]["Band"].as_str().expect("Value not found!").to_string();

        Ok(CellInfo {
            band,
            bars,
            strength
        })
    }

    // get the LTE metrics
    pub fn get_4g_stats(&self) -> ClientResult<CellInfo> {
        let response = self.get("fastmile_radio_status_web_app.cgi", AuthMode::None)?;
        let json = parse_json(&response)?;
        let bars = json["cell_5G_stats_cfg"][0]["stat"]["RSRPStrengthIndexCurrent"].as_u8().expect("Value not found!");
        let strength = json["cell_5G_stats_cfg"][0]["stat"]["RSRPCurrent"].as_i32().expect("Value not found!");
        let band = json["cell_LTE_stats_cfg"][0]["stat"]["Band"].as_str().expect("Value not found!").to_string();
        
        Ok(CellInfo {
            band,
            bars,
            strength
        })
    }

    // get the connection status as a boolean
    pub fn get_conn_status(&self) -> bool {
        let response = self.get("fastmile_radio_status_web_app.cgi",  AuthMode::None);
        if response.is_err() {
            return false;
        }
        let json_res = parse_json(&response.unwrap());

        if let Ok(json) = json_res {
            let val = json["connection_status"][0]["ConnectionStatus"].as_usize();
            if val.is_none() {
                return false;
            }
    
            return val.unwrap() == 1
        }
        false
    }

    pub fn get_all_stats(&self) -> ClientResult<GatewayStats> {
        let conn_status = self.get_conn_status();
        let info_5g = self.get_5g_stats()?;
        let info_4g = self.get_4g_stats()?;
        let download = self.get_download()?;
        let upload = self.get_upload()?;
        let sig_tracker = tracker::SignalTracker::get();
        let networks = self.get_networks()?;
        let devices = self.get_devices()?;

        Ok(GatewayStats {
            conn_status,
            devices,
            download,
            upload,
            info_4g,
            info_5g,
            networks,
            sig_tracker
        })
    }

    pub fn get_adv_stats(&self) -> ClientResult<AdvancedGatewayStats> {
        let response = self.get("fastmile_radio_status_web_app.cgi", AuthMode::None)?;
        let json = parse_json(&response)?;

        // APN info
        let apn_name = json["apn_cfg"][0]["APN"].as_str().unwrap().to_string();
        let apn_ip4 = json["apn_cfg"][0]["X_ALU_COM_IPAddressV4"].as_str().unwrap().to_string();
        let apn_ip6 = json["apn_cfg"][0]["X_ALU_COM_IPAddressV6"].as_str().unwrap().to_string();

        // 5G cell info
        let band_5g = json["cell_5G_stats_cfg"][0]["stat"]["Band"].as_str().unwrap().to_string();
        let rsrp_5g = json["cell_LTE_stats_cfg"][0]["stat"]["RSRPCurrent"].as_i32().unwrap();
        let snr_5g = json["cell_5G_stats_cfg"][0]["stat"]["SNRCurrent"].as_i32().unwrap();
        let rsrq_5g = json["cell_LTE_stats_cfg"][0]["stat"]["RSRQCurrent"].as_i32().unwrap();
        let rssi = json["cell_LTE_stats_cfg"][0]["stat"]["RSSICurrent"].as_i32().unwrap();
        
        // 4G cell info
        let band_4g = json["cell_LTE_stats_cfg"][0]["stat"]["Band"].as_str().unwrap().to_string();
        let rsrp_4g = json["cell_5G_stats_cfg"][0]["stat"]["RSRPCurrent"].as_i32().unwrap();
        let snr_4g = json["cell_LTE_stats_cfg"][0]["stat"]["SNRCurrent"].as_i32().unwrap();
        let rsrq_4g = json["cell_5G_stats_cfg"][0]["stat"]["RSRQCurrent"].as_i32().unwrap();

        // SIM info
        let response = self.get("fastmile_statistics_status_web_app.cgi", AuthMode::Web)?;
        let sim_json = parse_json(&response)?;
        let imsi = sim_json["sim_cfg"][0]["IMSI"].as_str().unwrap().parse().unwrap();
        let iccid = sim_json["sim_cfg"][0]["ICCID"].as_str().unwrap().parse().unwrap();
        let imei = sim_json["network_cfg"][0]["IMEI"].as_str().unwrap().parse().unwrap();

        Ok(AdvancedGatewayStats {
            apn_name,
            apn_ip4,
            apn_ip6,
            band_5g,
            rsrp_5g,
            snr_5g,
            rsrq_5g,
            rssi,
            band_4g,
            rsrp_4g,
            snr_4g,
            rsrq_4g,
            imsi,
            iccid,
            imei
        })
    }

    // gets info about the gateway
    pub fn get_info(&self) -> ClientResult<GatewayInfo> {
        let response = self.get("device_status_web_app.cgi", AuthMode::Web)?;
        let json = parse_json(&response)?;
        let response = self.get("fastmile_statistics_status_web_app.cgi", AuthMode::Web)?;
        let sim_json = parse_json(&response)?;
        let vendor = json["Vendor"].as_str().unwrap().to_string();
        let ser_num = json["SerialNumber"].as_str().unwrap().to_string();
        let hw_ver = json["HardwareVersion"].as_str().unwrap().to_string();
        let sw_ver = json["SoftwareVersion"].as_str().unwrap().to_string();
        let uptime = json["UpTime"].as_usize().unwrap();
        let imei = sim_json["network_cfg"][0]["IMEI"].as_str().unwrap().parse().unwrap();
        let info = GatewayInfo {
            vendor,
            ser_num,
            hw_ver,
            sw_ver,
            uptime,
            imei
        };
        Ok(info)
    }

    pub fn reload_config(&mut self) {
        self.config = GlobalConfig::load();
        self.gw_config = GatewayConfig::load();
        self.dev_config = DeviceConfig::load();
    }

    pub fn is_valid_net(&self) -> bool {
        is_valid_net(&self.config.valid_networks)
    }

    // POST request to the gateway
    fn post(&self, res: &str, mode: AuthMode) -> ClientResult<()> {
        let addr = format!("http://{}/{}", self.config.gateway_ip, res);
        let req = self.client.post(addr);
        let req_r = match mode {
            /*AuthMode::App => {
                let _token = self.gen_app_token()?;
                req.send()
            },*/
            AuthMode::Web => {
                let token = self.gen_web_token()?;
                req.form(&token.get_data()).send()
            },
            AuthMode::None => req.send()
        };

        match req_r {
            Ok(_) => Ok(()),
            Err(why) => Err(ClientError::GatewayUnreachable(why.to_string()))
        }
    }

    // GET request to the gateway
    fn get(&self, res: &str, mode: AuthMode) -> ClientResult<String> {
        let addr = format!("http://{}/{}", self.config.gateway_ip, res);
        let req = self.client.get(addr);
        let req_r = match mode {
            /*AuthMode::App => {
                let _token = self.gen_app_token()?;
                req.send()
            },*/
            AuthMode::Web => {
                let _token = self.gen_web_token()?;
                req.send()
            },

            AuthMode::None => req.send()
        };

        match req_r {
            Ok(response) => Ok(response.text().unwrap()),
            Err(why) => Err(ClientError::GatewayUnreachable(why.to_string()))
        }
    }
    
    fn gen_web_token(&self) -> ClientResult<AuthToken> {
        let token_r = AuthToken::get_web(&self.client, &self.config.username, &self.config.password);
        if let Ok(token) = token_r {
            token.put_cookies_in_jar(&self.jar);
            Ok(token)
        }
        else {
            token_r
        }
    }

    /*fn gen_app_token(&self) -> Result<AuthToken, ClientError> {
        let token_r = AuthToken::get_app(&self.client, &self.config.username, &self.config.password);
        if let Ok(token) = token_r {
            token.put_cookies_in_jar(&self.jar);
            Ok(token)
        }
        else {
            token_r
        }
    }*/

    fn get_upload(&self) -> ClientResult<u128> {
        let response = self.get("fastmile_radio_status_web_app.cgi", AuthMode::None)?;
        let json = parse_json(&response)?;
        Ok(json["cellular_stats"][0]["BytesSent"].as_u64().expect("Value not found!") as u128)
    }

    fn get_download(&self) -> ClientResult<u128> {
        let response = self.get("fastmile_radio_status_web_app.cgi", AuthMode::None)?;
        let json = parse_json(&response)?;
        Ok(json["cellular_stats"][0]["BytesReceived"].as_u64().expect("Value not found!") as u128)
    }

    fn get_networks(&self) -> ClientResult<Vec<NetworkInfo>> {
        let response = self.get("statistics_status_web_app.cgi", AuthMode::None)?;
        let json = parse_json(&response)?;
        let eth_networks = json["LAN"].members();
        let mut networks: Vec<NetworkInfo> = vec![];
        for (i, network_raw) in eth_networks.enumerate() {
            let id = network_raw["Name"].as_str().expect("Value not found!").to_string();
            let name = format!("Ethernet {}", i + 1);
            let upload = network_raw["BytesSent"].as_u64().expect("Value not found!") as u128;
            let download = network_raw["BytesReceived"].as_u64().expect("Value not found!") as u128;
            let enable_str = &network_raw["Enable"].as_u32().expect("Value not found!");

            networks.push(NetworkInfo {
                id,
                name,
                upload,
                download,
                enabled: (*enable_str == 1)
            })
        }

        let wifi_networks = json["WLAN"].members();

        for network_raw in wifi_networks {
            let id = network_raw["Name"].as_str().expect("Value not found!").to_string();
            let name = network_raw["SSID"].as_str().expect("Value not found!").to_string();
            let upload = network_raw["BytesSent"].as_u64().expect("Value not found!") as u128;
            let download = network_raw["BytesReceived"].as_u64().expect("Value not found!") as u128;
            let enable = &network_raw["Enable"].as_u32().expect("Value not found!");

            networks.push(NetworkInfo {
                id,
                name,
                upload,
                download,
                enabled: (*enable == 1)
            })
        }

        Ok(networks)
    }
}

// parse text as json
fn parse_json(source: &str) -> ClientResult<json::JsonValue> {
    match json::parse(source) {
        Ok(j) => Ok(j),
        Err(why) => return Err(ClientError::JSONParseError(why.to_string(), source.to_string()))
    }
}

// get the dates of all the stored logs
pub fn get_log_dates() -> Vec<(u32, u32, u32)> {
    let files = utils::get_dir(&*LOG_DIR, false);
    let mut dates = Vec::new();
    for file in files.iter().rev() {
        if file.contains("gatewaymon") {
            let mut split1 = file.split(".");
            let mut split2 = split1.nth(0).unwrap().split("-");
            split2.next();
            dates.push((
                split2.next().unwrap().parse().unwrap(),
                split2.next().unwrap().parse().unwrap(),
                split2.next().unwrap().parse().unwrap()
            ));
        }
    }
    let mut uniques = HashSet::new();
    dates.retain(|d| uniques.insert(*d));
    dates.sort();
    dates.reverse();
    dates
}

// get a log file by date
pub fn get_log(date: (u32, u32, u32)) -> StyledString {
    let raw = get_log_plain(date);
    let mut styled = StyledString::new();
    for line in raw.lines() {
        let style = if line.contains("DEBUG]") {
            ColorStyle::from(Color::Dark(BaseColor::Cyan))
        }
        else if line.contains("WARN]") {
            ColorStyle::from(Color::Light(BaseColor::Yellow))
        }
        else if line.contains("ERROR]") {
            ColorStyle::from(Color::Light(BaseColor::Red))
        }
        else if line.contains("FATAL]") {
            ColorStyle::from((Color::Light(BaseColor::Red), Color::Dark(BaseColor::Black)))
        }
        else {
            ColorStyle::from(Color::Dark(BaseColor::Green))  
        };
        styled.append_styled(line, style);
        styled.append("\n");
    }
    styled
}

pub fn get_log_plain(date: (u32, u32, u32)) -> String {
    fs::read_to_string(format!("{}/gatewaymon-{}-{}-{}.log", *LOG_DIR, date.0, date.1, date.2)).unwrap_or(" ".to_string())
}

// check the network environment
// 0: network is fine
// 1: gateway needs rebooted
// 2: network is disconnected
pub fn check_net_env(full: bool) -> usize {
    let output_str = if full {
        LOG.line(LogLevel::Debug(*DEBUG), "Performing full check of network environment...", true);
        if ping_req() {
            return 0;
        }
        else {
            nmcli(["-t", "-f", "CONNECTIVITY", "networking", "connectivity", "check"], true)
        }
    }
    else {
        nmcli(["-t", "-f", "CONNECTIVITY", "general"], true)
    };
    if output_str.contains("none") { 2 }
    else if output_str.contains("full") { 0 }
    else { 1 }
}

// are we connected to a valid network ?
pub fn is_valid_net(valid_networks: &Vec<String>) -> bool {
    let networks = connected_networks();
    for network in networks {
        if network.1.contains("ethernet") {
            return true;
        }

        for name in valid_networks {
            if network.0 == *name {
                return true;
            }
        }
    }

    false
}

// nmcli network list
pub fn connected_networks() -> Vec<(String, String)> {
    let out = nmcli(["-t", "-f", "NAME,TYPE", "con", "show", "--active"], true);
    out.split("\n").map(|line: &str| {
        let vals: Vec<&str> = line.split(":").collect();
        (vals[0].to_string(), vals[1].to_string())
    }).collect()
}

// execute a nmcli command
pub fn nmcli<A: IntoIterator<Item = S>, S: AsRef<OsStr>>(args: A, print_stdout: bool) -> String {
    match Command::new("nmcli").args(args).output() {
        Ok(o) => {
            let success = o.status.success();
            let mut o_fmt = if success { String::from_utf8(o.stdout).unwrap() } else { String::from_utf8(o.stderr).unwrap() };
            o_fmt.pop();
            if !success {
                for line in o_fmt.to_string().lines() {
                    LOG.line(LogLevel::Error, line, print_stdout)
                }
                LOG.line(LogLevel::Warn, "nmcli command failed to execute properly!", true);
            }
            o_fmt
        },

        Err(why) => {
            for line in why.to_string().lines() {
                LOG.line(LogLevel::Error, line, true);
            }
            LOG.line(LogLevel::Warn, "nmcli command failed to execute properly!", true);
            why.to_string()
        }
    }
}

// perform a ping request
// true: the ping was successful
// false: there a was problem
fn ping_req() -> bool {
    let mut ping_cmd = Command::new("ping");
    ping_cmd.args(["paradisemod.net", "-c1", "-w10"]);
    let out = ping_cmd.output().unwrap();

    if *DEBUG {
        let dbg_txt = if out.status.success() { String::from_utf8(out.stdout).unwrap() } else { String::from_utf8(out.stderr).unwrap() };
        LOG.line(LogLevel::Debug(true), "Ping output:", *PRINT_STDOUT);
        for line in dbg_txt.lines() {
            LOG.line(LogLevel::Debug(true), line, *PRINT_STDOUT);
        }
    }
    out.status.success()
}