use std::{
    sync::{Arc, RwLock},
    time::Duration,
    thread, env,
    collections::HashMap
};
use cursive::{
    menu,
    event::{Event, Key},
    theme::{PaletteColor, Color, BaseColor, ColorStyle},
    views::{
        Dialog,
        LinearLayout,
        ListView,
        OnEventView,
        TextView,
        EditView,
        NamedView,
        Button,
        SelectView,
        ResizedView
    },
    utils::markup::StyledString,
    view::{Nameable, Scrollable},
    traits::*
};
use colored::Colorize;
use rust_utils::{
    logging::LogLevel,
    config::Config
};
use cursive_extras::*;
use crate::{
    client::{self, TrashcanClient},
    stats::*,
    hosts::*,
    LOG,
};

mod views;
use views::*;

type AtomicClient = Arc<RwLock<TrashcanClient>>;
const MESSAGES: [&'static str; 3] = [
    "Network seems fine...",
    "Your gateway needs rebooting!",
    "Gateway is off or rebooting"
];

pub fn init() {
    println!("Collecting gateway statistics...");
    let client_raw = match TrashcanClient::new() {
        Ok(c) => c,
        Err(why) => {
            let msg = format!("Unable to connect to gateway: {}", why).bright_red();
            eprintln!("{}", msg);
            return;
        }
    };

    let client = Arc::new(RwLock::new(client_raw));
    let worker_client = client.clone();
    let ui_client = client.clone();
    let stats = match client.read().unwrap().get_all_stats() {
        Ok(s) => s,
        Err(why) => {
            let msg = format!("Unable to get gateway stats: {}", why).bright_red();
            eprintln!("{}", msg);
            return;
        }
    };

    let view_stats = Arc::new(RwLock::new(stats.clone()));
    let worker_stats = view_stats.clone();
    let menu_stats = view_stats.clone();

    // background thread that updates the stats
    thread::spawn(move || {
        loop {
            thread::sleep(Duration::from_secs(1));
            let mut stats = match worker_stats.write() {
                Ok(s) => s,
                Err(why) => {
                    LOG.line(LogLevel::Error, why, false);
                    continue;
                }
            };
            *stats = match worker_client.read().unwrap().get_all_stats() {
                Ok(s) => s,
                Err(why) => {
                    LOG.line(LogLevel::Error, why, false);
                    continue;
                }
            };
        }
    });

    let mut theme = better_theme();
    theme.palette[PaletteColor::Highlight] = Color::Light(BaseColor::Magenta);
    theme.palette[PaletteColor::HighlightInactive] = Color::Dark(BaseColor::Magenta);
    
    let mut root = cursive::default();
    root.set_user_data(ui_client);

    // menu
    root.set_autohide_menu(false);
    root.add_global_callback(Key::Up, |v| v.select_menubar());
    root.menubar()
        .add_subtree("Settings", menu::Tree::new()
            .leaf("General Settings", |view| {
                let mut status = view.find_name::<StatusView>("status").unwrap();
                let client: &AtomicClient = view.user_data().unwrap();
                let config = report_error!(status, client.read()).config.clone();
                view.add_layer(
                    settings!(
                        "General Settings",
                        move |view| {
                            let mut status = view.find_name::<StatusView>("status").unwrap();
                            let mut config2 = config.clone();
                            let ip_field = view.find_name::<EditView>("ip").unwrap();
                            let username_field = view.find_name::<EditView>("username").unwrap();
                            let password_field = view.find_name::<EditView>("password").unwrap();
                            let net_list = view.find_name::<OnEventView<SelectView<String>>>("net_list").unwrap();
                            config2.valid_networks.clear();
                            for net in net_list.get_inner().iter() {
                                config2.valid_networks.push(net.1.clone());
                            }
                            config2.gateway_ip = ip_field.get_content().to_string();
                            config2.username = username_field.get_content().to_string();
                            config2.password = password_field.get_content().to_string();
                            report_error!(status, config2.save());
                            status.info("Settings saved");
                            view.pop_layer();
                        },
                        TextView::new("Gateway IP address:"),
                        styled_editview(&config.gateway_ip, "ip", false),
                        TextView::new("Password:"),
                        styled_editview(&config.password, "password", true),
                        Button::new("Show password", |view| {
                            let mut password = view.find_name::<EditView>("password").unwrap();
                            password.set_secret(false);
                        }),
                        Button::new("Hide password", |view| {
                            let mut password = view.find_name::<EditView>("password").unwrap();
                            password.set_secret(true);
                        }),
                        TextView::new("\nKnown Networks:"),
                        known_network_list(&config.valid_networks).scrollable(),
                        TextView::new("\nAdd network name:"),
                        styled_editview("", "new_net", false),
                        Button::new("Add", |view| {
                            let new_net = view.find_name::<EditView>("new_net").unwrap();
                            let mut net_list = view.find_name::<OnEventView<SelectView<String>>>("net_list").unwrap();
                            net_list.get_inner_mut().add_item_str(new_net.get_content().to_string());
                        })
                    )
                );
                
            })
            .leaf("Host Settings", |view| {
                let mut status = view.find_name::<StatusView>("status").unwrap();
                let client: &AtomicClient = view.user_data().unwrap();
                let config = report_error!(status, client.read()).hst_config.clone();
                view.add_layer(
                    settings!(
                        "Host Settings",
                        move |view| {
                            let mut status = view.find_name::<StatusView>("status").unwrap();
                            let mut config2 = config.clone();
                            config2.aliases.clear();
                            let tld_field = view.find_name::<EditView>("tld").unwrap();
                            let name_field = view.find_name::<EditView>("name").unwrap();
                            let host_list = view.find_name::<OnEventView<SelectView<(String, String)>>>("host_list").unwrap();
                            for host in host_list.get_inner().iter() {
                                config2.aliases.insert(host.1.0.to_string(), host.1.1.to_string());
                            }
                            config2.tld = tld_field.get_content().to_string();
                            config2.gateway_name = name_field.get_content().to_string();
                            report_error!(status, config2.save());
                            status.info("Settings saved");
                            view.pop_layer();
                        },
                        TextView::new("Top level domain:"),
                        styled_editview(&config.tld, "tld", false),
                        TextView::new("Gateway name:"),
                        styled_editview(&config.gateway_name, "name", false),
                        TextView::new("Host Aliases:"),
                        host_alias_list(&config.aliases)
                    )
                );
            })
            .leaf("Gateway Settings", |view| {
                let mut status = view.find_name::<StatusView>("status").unwrap();
                let client: &AtomicClient = view.user_data().unwrap();
                let config = report_error!(status, client.read()).gw_config.clone();
                let roe_label = format!("Reboot on network issues: {}", if config.reboot_on_err { "Yes" } else { "No" });
                let ar_label = format!("Auto reboot: {}", if config.auto_reboot { "Yes" } else { "No" });
                view.add_layer(
                    settings!(
                        "Gateway Settings",
                        move |view| {
                            let mut status = view.find_name::<StatusView>("status").unwrap();
                            let ar = view.find_name::<Button>("ar").unwrap();
                            let roe = view.find_name::<Button>("roe").unwrap();
                            let rhr = view.find_name::<EditView>("rhr").unwrap();
                            let rmin = view.find_name::<EditView>("rmin").unwrap();
                            let mut config2 = config.clone();
                            config2.reboot_on_err = roe.label() == "Reboot on network issues: Yes";
                            config2.auto_reboot = ar.label() == "Auto reboot: Yes";
                            config2.reboot_hr = report_error!(status, rhr.get_content().parse());
                            config2.reboot_min = report_error!(status, rmin.get_content().parse());
                            report_error!(status, config2.save());
                            status.info("Settings saved");
                            view.pop_layer();
                        },
                        Button::new_raw(roe_label, |view| {
                            let mut button = view.find_name::<Button>("roe").unwrap();
                            if button.label() == "Reboot on network issues: Yes" {
                                button.set_label_raw("Reboot on network issues: No");
                            }
                            else {
                                button.set_label_raw("Reboot on network issues: Yes");
                            }
                        }).with_name("roe"),
                        Button::new_raw(ar_label, |view| {
                            let mut button = view.find_name::<Button>("ar").unwrap();
                            if button.label() == "Auto reboot: Yes" {
                                button.set_label_raw("Auto reboot: No");
                            }
                            else {
                                button.set_label_raw("Auto reboot: Yes");
                            }
                        }).with_name("ar"),
                        hlayout!(
                            TextView::new("Reboot Time (24 hour time):"),
                            ResizedView::with_fixed_size((3, 1), styled_editview(&config.reboot_hr.to_string(), "rhr", false)),
                            TextView::new(":"),
                            ResizedView::with_fixed_size((3, 1), styled_editview(&format!("{:02}",config.reboot_min), "rmin", false))
                        )
                    )
                );
            })
            .leaf("Reboot Gateway",|view| {
                view.add_layer(
                    confirm_dialog("Reboot Gateway", "Are you sure? Your network will be down for about 5 minutes.", |view| { 
                        let mut status = view.find_name::<StatusView>("status").unwrap();
                        let client: &AtomicClient = view.user_data().unwrap();
                        report_error!(status, client.write().unwrap().reboot_gateway(false));
                        view.quit();
                        println!("Rebooting gateway...");
                    })
                )
            })
        )
        .add_subtree("Diagnostics", menu::Tree::new()
            .leaf("Check Network for Problems", |view| {
                let mut status = view.find_name::<StatusView>("status").unwrap();
                let message = MESSAGES[client::check_net_env(true)];
                status.info(message);
            })
            .leaf("Gateway Monitor Log", |view| {
                let mut status = view.find_name::<StatusView>("status").unwrap();
                let dates = client::get_log_dates();
                if dates.len() == 0 {
                    status.info("Gateway Monitor Log is empty!");
                    return;
                }
                let mut date_selector = SelectView::new()
                    .on_select(move |view, date| {
                        let mut log = view.find_name::<TextView>("log").unwrap();
                        log.set_content(client::get_log(*date));
                    });

                    for date in &dates {
                        date_selector.add_item(format!("{:02}/{:02}/{:04}", date.1, date.2, date.0), *date)
                    }
    
                    let date = dates.get(0);
                    view.add_layer(
                        Dialog::around(
                            ResizedView::with_max_height(18, 
                                hlayout!(
                                    date_selector.scrollable(),
                                    HDivider::new(),
                                    get_log(*date.unwrap_or(&(2020, 3, 1))).with_name("log").scrollable().max_size((50, 40))
                                )
                            )
                        )
                        .dismiss_button("Back")
                        .title("Gateway Monitor Log")
                        .wrap_with(OnEventView::new)
                        .on_event(Event::Key(Key::Esc), |v| { v.pop_layer(); })
                    )
            })
            .leaf("Connection Stats Log", move |view| {
                let mut status = view.find_name::<StatusView>("status").unwrap();
                let stats = menu_stats.read().unwrap();
                let sig_tracker = stats.sig_tracker.clone();
                let dates = sig_tracker.get_dates();
                let first_date =  match dates.get(0) {
                    Some(v) => v,
                    None => {
                        status.info("Connection Stats Log is empty!");
                        return;
                    }
                };
                let sv_sig_tracker = sig_tracker.clone();
                let entries = sig_tracker.get_formatted_entries(*first_date);
                let mut date_selector: SelectView<(u32, u32, u32)> = SelectView::new()
                    .on_select(move |view, date| {
                        let mut table = view.find_name::<LinearLayout>("table").unwrap();
                        let entries = sv_sig_tracker.get_formatted_entries(*date);
                        *table = get_table(entries);
                    });
                for date in dates {
                    date_selector.add_item(format!("{:02}/{:02}/{:04}", date.1, date.2, date.0), date)
                }

                view.add_layer(
                    Dialog::around(
                        hlayout!(
                            date_selector.scrollable(),
                            get_table(entries).with_name("table")
                        )
                    )
                    .dismiss_button("Back")
                    .title("Connection Stats Log")
                    .wrap_with(OnEventView::new)
                    .on_event(Event::Key(Key::Esc), |v| { v.pop_layer(); })
                )
            })
            .leaf("Connection Stats Visual", move |view| {
                let mut status = view.find_name::<StatusView>("status").unwrap();
                let sig_tracker = stats.sig_tracker.clone();
                let dates = sig_tracker.get_dates();
                if dates.len() == 0 {
                    status.info("Connection Stats Log is empty!");
                    return;
                }
                let sv_sig_tracker = sig_tracker.clone();
                let mut date_selector: SelectView<(u32, u32, u32)> = SelectView::new()
                    .on_select(move |view, date| {
                        let mut graph = view.find_name::<ConnStatVisual>("graph").unwrap();
                        let mut new_entries = Vec::new();
                        for entry in &sv_sig_tracker.entries {
                            if entry.date == *date {
                                new_entries.push((entry.info_5g.strength, entry.info_4g.strength, entry.time.0, entry.reboot))
                            };
                        }

                        *graph = ConnStatVisual::new(new_entries);
                    });
                for date in &dates {
                    date_selector.add_item(format!("{:02}/{:02}/{:04}", date.1, date.2, date.0), *date)
                }

                let mut new_entries = Vec::new();
                for entry in &sig_tracker.entries {
                    if entry.date == dates[0] {
                        new_entries.push((entry.info_5g.strength, entry.info_4g.strength, entry.time.0, entry.reboot));
                    }
                }

                let style_5g = ColorStyle::new(Color::TerminalDefault, Color::Light(BaseColor::Cyan));
                let style_5g_r = ColorStyle::new(Color::TerminalDefault, Color::Light(BaseColor::Red));
                let style_4g = ColorStyle::new(Color::TerminalDefault, Color::Dark(BaseColor::Cyan));
                let style_4g_r = ColorStyle::new(Color::TerminalDefault, Color::Dark(BaseColor::Red));

                let mut legend = StyledString::from("Legend:\n");
                legend.append(StyledString::styled(" ", style_5g));
                legend.append(StyledString::styled(" ", style_5g_r));
                legend.append(" 5G Signal\n");
                legend.append(StyledString::styled(" ", style_4g));
                legend.append(StyledString::styled(" ", style_4g_r));
                legend.append(" 4G (LTE) Signal\n\nRed indicates the gateway was rebooted");

                view.add_layer(
                    Dialog::around(
                        ResizedView::with_max_height(21,
                            hlayout!(
                                date_selector.scrollable(),
                                HDivider::new(),
                                vlayout!(
                                    hlayout!(
                                        ConnStatVisual::get_info_view(),
                                        ResizedView::with_max_size((41, 16),
                                            ConnStatVisual::new(new_entries)
                                                .with_name("graph")
                                                .scrollable()
                                                .scroll_x(true)
                                        )
                                    ),
                                    TextView::new(legend)
                                )
                            )
                        )
                    )
                    .title("Connection Stats Visual")
                    .dismiss_button("Back")
                    .wrap_with(OnEventView::new)
                    .on_event(Event::Key(Key::Esc), |v| { v.pop_layer(); })
                )
            })
        )
        .add_leaf("Gateway Info", |view| {
            let mut status = view.find_name::<StatusView>("status").unwrap();
            let client: &AtomicClient = view.user_data().unwrap();
            let info = report_error!(status, client.read().unwrap().get_info());
            view.add_layer(info_dialog("Gateway Info", info));
        })
        .add_leaf("Help", |view| {
            let log_path = format!("{}/.local/share/tmo-tools", env::var("HOME").expect("Where the hell is your home folder?!"));
            let cfg_path = format!("{}/.config/tmo-tools", env::var("HOME").expect("Where the hell is your home folder?!"));
            let help_text = format!("T-Mobile Home Internet Tools (tmotop)\n\
            Version {}\n\
            \n\
            Key Shortcuts:\n\
            q: Quit\n\
            F5: Reload all settings\n\
            \n\
            Gateway web interface: http://192.168.12.1/web_whw/#/overview\n\
            Log file directory: {}\n\
            Settings directory: {}",env!("CARGO_PKG_VERSION"), log_path, cfg_path);
            view.add_layer(info_dialog("Help", help_text));
        });
    root.set_theme(theme);

    root.add_fullscreen_layer(
        vlayout!(
            hlayout!(
                Dialog::around(DataUsageView::new(stats.download, stats.upload).with_name("data_usage")).title("Data Usage"),
                Dialog::around(CellStatusView::new(&stats.info_4g, &stats.info_5g).with_name("cell_status")).title("Connection Status")
            ),
            hlayout!(
                Dialog::around(get_host_list(&stats.devices).with_name("devices").scrollable()).title("Connected Devices"),
                Dialog::around(get_network_list(&stats.networks).with_name("networks").scrollable()).title("Active Networks")
            ),
            StatusView::new().with_name("status")
        )
    );
    root.add_global_callback('q', |view| view.quit());

    // refresh the dashboard
    // the background thread auto updates the stats
    root.add_global_callback(Event::Refresh, move |view| {
        let mut data_usage = view.find_name::<DataUsageView>("data_usage").unwrap();
        let mut cell_status = view.find_name::<CellStatusView>("cell_status").unwrap();
        let mut devices = view.find_name::<ListView>("devices").unwrap();
        let mut networks = view.find_name::<ListView>("networks").unwrap();
        let mut status = view.find_name::<StatusView>("status").unwrap();

        if let Ok(stats) = view_stats.try_read() {
            // reload data usage stats
            data_usage.update(stats.download, stats.upload);
            
            // reload cell stats
            cell_status.update(&stats.info_4g, &stats.info_5g);

            // reload device list
            *devices = get_host_list(&stats.devices);

            // reload network list
            *networks = get_network_list(&stats.networks);
        }
        status.update();
    });

    // reload the config when F5 is pressed
    root.add_global_callback(Event::Key(Key::F5), move |view| {
        let mut status = view.find_name::<StatusView>("status").unwrap();
        status.info("Reloading settings...");
        status.update();
        client.write().unwrap().reload_config();
        status.info("Reloaded settings");
    });

    root.set_fps(30);
    root.run();
}

fn get_table(data: (StyledString, StyledString, StyledString)) -> LinearLayout {
    vlayout!(
        TextView::new(data.0).no_wrap(),
        TextView::new(data.1).no_wrap().scrollable(),
        TextView::new(data.2).no_wrap()
    )
}

fn get_log(date: (u32, u32, u32)) -> TextView {
    TextView::new(client::get_log(date))
}

fn get_host_list(hosts: &HostList) -> ListView {
    let mut view = ListView::new();
    for host in &hosts.hosts {
        let dev_view = ConnectedDeviceView::new(host.clone());
        view.add_child("", dev_view);
    }
    view
}

fn get_network_list(networks: &Vec<NetworkInfo>) -> ListView {
    let mut view = ListView::new();
    for network in networks {
        let dev_view = NetworkView::new(network.clone());
        if network.enabled {
            view.add_child("", dev_view);
        }
    }
    view
}

fn known_network_list(networks: &Vec<String>) -> NamedView<OnEventView<SelectView<String>>> {
    let mut list: SelectView<String> = SelectView::new();
    for network in networks {
        list.add_item_str(network)
    }
    list
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Del), |view| {
            view.add_layer(confirm_dialog("", "Are you sure?", |view| {
                let mut nl_t = view.find_name::<OnEventView<SelectView<String>>>("net_list").unwrap();
                let nlist = nl_t.get_inner_mut();
                let idx = nlist.selected_id().unwrap();
                nlist.remove_item(idx);
                view.pop_layer();
            }));
        }).with_name("net_list")
}

fn host_alias_list(aliases: &HashMap<String, String>) -> NamedView<OnEventView<SelectView<(String, String)>>> {
    let mut list: SelectView<(String, String)> = SelectView::new();
    for alias in aliases {
        list.add_item(format!("{} : {}", alias.0, alias.1), (alias.0.clone(), alias.1.clone()));
    }

    list
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Del), |view| {
            view.add_layer(confirm_dialog("", "Are you sure?", |view| {
                let mut hl_t = view.find_name::<OnEventView<SelectView<(String, String)>>>("host_list").unwrap();
                let hlist = hl_t.get_inner_mut();
                let idx = hlist.selected_id().unwrap();
                hlist.remove_item(idx);
                view.pop_layer();
            }));
        })
    .with_name("host_list")
}