use std::{
    process, thread,
    time::{Instant, Duration},
};
use chrono::{Timelike, Datelike, Local};
use rust_utils::{
    config::Config,
    logging::LogLevel
};
use crate::client::{self, tracker::*,};
use crate::config::GlobalConfig;
use crate::{LOG, DEBUG};

// gateway monitoring daemon (background process)
// this will check the network environment once every minute and will tell the gateway to reboot if there are any problems
pub fn init() {
    LOG.line_basic("Starting T-Mobile Gateway Monitor...", true);
    let mut client = match client::TrashcanClient::new() {
        Ok(c) => c,
        Err(error) => {
            if !client::is_valid_net(&GlobalConfig::load().valid_networks) {
                LOG.line(LogLevel::Error, "This network is not a T-Mobile home Internet network", true);
            }
            else {
                LOG.line(LogLevel::Error, format!("Error starting T-Mobile Gateway Monitor: {}", error), true);
            }
            return;
        }
    };

    let mut tracker = SignalTracker::get();

    LOG.line_basic("Startup complete.", true);
    LOG.line(LogLevel::Debug(*DEBUG), "Debug logging enabled", true);

    ctrlc::set_handler(|| {
        LOG.line_basic("Shutting down...", true);
        process::exit(0);
    }).expect("Error setting Ctrl-C handler");

    let mut startup = true;
    let mut reboot_time = Instant::now();
    let mut collect = true;
    loop {
        let now = Local::now();
        
        // if network issues are encountered, request the gateway to restart
        // if not configured to reboot the gateway, just reset the wifi connection
        if client.is_valid_net() {
            if client::check_net_env(now.second() == 0) != 0 {
                LOG.line(LogLevel::Info, "Why the hell is the gateway causing issues again?!", true);
                if !startup && reboot_time.elapsed().as_secs() <= 300 {
                    LOG.line(LogLevel::Warn, "You may want to unplug your gateway", true);
                }
                let stats_4g = client.get_4g_stats();
                let stats_5g = client.get_5g_stats();
                let conn_stat = client.get_conn_status();

                if client.gw_config.reboot_on_err {
                    match client.reboot_gateway(true) {
                        Ok(()) => {
                            reboot_time = Instant::now();
                            startup = false;
                            LOG.line_basic("Waiting for network to come online again...", true);
                            reset_wifi();

                            // wait for the network to come back online
                            while client::check_net_env(true) == 2 { }
                            thread::sleep(Duration::from_secs(15));
                            let stats_5g_u = match stats_5g {
                                Ok(stats) => stats,

                                Err(why) => {
                                    LOG.line(LogLevel::Error, format!("Error getting signal stats: {}", why), true);
                                    continue;
                                }
                            };

                            let stats_4g_u = match stats_4g {
                                Ok(stats) => stats,

                                Err(error) => {
                                    LOG.line(LogLevel::Error, format!("Error getting signal stats: {}", error), true);
                                    continue;
                                }
                            };

                            tracker.add_entry((now.hour(), now.minute()), (now.year() as u32, now.month(), now.day()), stats_5g_u, stats_4g_u, conn_stat, true);
                            LOG.line_basic("Gateway has been rebooted", true);
                        },

                        Err(error) => LOG.line(LogLevel::Error, format!("Error rebooting gateway: {}", error), true)
                    };
                }
                else {
                    reset_wifi();
                    thread::sleep(Duration::from_secs(15));
                }
            }

            // request the gateway to reboot at a specified time
            if now.hour() == client.gw_config.reboot_hr && now.minute() == client.gw_config.reboot_min && client.gw_config.auto_reboot {
                match client.reboot_gateway(true) {
                    Ok(()) => {
                        LOG.line_basic("Waiting for network to come online again...", true);
                        reset_wifi();

                        // wait for the network to come back online
                        while client::check_net_env(true) == 2 { }
                        LOG.line(LogLevel::Info, "Gateway has been rebooted", true);
                        thread::sleep(Duration::from_secs(15));
                    },

                    Err(error) => LOG.line(LogLevel::Error, format!("Error rebooting gateway: {}", error), true)
                };
            }

            if now.minute() % 15 == 0 && collect {
                let stats_4g = client.get_4g_stats();
                let stats_5g = client.get_5g_stats();
                let conn_stat = client.get_conn_status();
                let stats_5g_u = match stats_5g {
                    Ok(stats) => stats,

                    Err(error) => {
                        LOG.line(LogLevel::Error, format!("Error getting signal stats: {}", error), true);
                        continue;
                    }
                };

                let stats_4g_u = match stats_4g {
                    Ok(stats) => stats,

                    Err(error) => {
                        LOG.line(LogLevel::Error, format!("Error getting signal stats: {}", error), true);
                        continue;
                    }
                };
                tracker.add_entry((now.hour(), now.minute()), (now.year() as u32, now.month(), now.day()), stats_5g_u, stats_4g_u, conn_stat, false);
                collect = false
            }
            else if now.minute() % 15 != 0 {
                collect = true;
            }
        }
        else {
            LOG.line(LogLevel::Warn, "This is not a T-Mobile Home Internet network", true);
            LOG.line(LogLevel::Warn, "Shutting down...", true);
            process::exit(1);
        }
    }
}

// turn off wifi for 30 seconds and reenable again
fn reset_wifi() {
    client::nmcli(["radio", "wifi", "off"], true);
    thread::sleep(Duration::from_secs(30));
    client::nmcli(["radio", "wifi", "on"], true);
}