use serde_derive::{Serialize, Deserialize};
use crate::{
    client::tracker::SignalTracker,
    hosts::HostList
};
use std::fmt;
use cursive::{
    utils::markup::StyledString,
    theme::{ColorStyle, BaseColor, Color}
};
use pad::*;

#[derive(Clone)]
pub struct GatewayStats {
    pub conn_status: bool,
    pub download: u128,
    pub upload: u128,
    pub info_5g: CellInfo,
    pub info_4g: CellInfo,
    pub devices: HostList,
    pub sig_tracker: SignalTracker,
    pub networks: Vec<NetworkInfo>,
}

#[derive(Clone)]
pub struct NetworkInfo {
    pub id: String,
    pub name: String,
    pub download: u128,
    pub upload: u128,
    pub enabled: bool
}

#[derive(Serialize, Deserialize, Clone)]
pub struct CellInfo {
    pub band: String,
    pub strength: i32,
    pub bars: u8
}

impl CellInfo {
    pub fn fmt_colored(&self) -> StyledString {
        let mut out_str = StyledString::plain(format!("Band: {} Strength: {} dBm ", self.band, self.strength).with_exact_width(29));
        out_str.append_styled(format!("{: <5}", "█".repeat(self.bars as usize)), ColorStyle::from((Color::Light(BaseColor::Magenta), Color::TerminalDefault)));
        out_str.append(" ".repeat(34 - out_str.source().chars().count()));
        out_str
    }
}

#[derive(Clone)]
pub struct GatewayInfo {
    pub vendor: String,
    pub ser_num: String,
    pub hw_ver: String,
    pub sw_ver: String,
    pub uptime: usize
}

impl fmt::Display for GatewayInfo {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        let ut_secs = self.uptime % 60;
        let ut_mins = (self.uptime - (self.uptime % 60)) / 60;
        let ut_hrs = ut_mins / 60;

        writeln!(f, "Vendor: {}", self.vendor)?;
        writeln!(f, "Serial Number: {}", self.ser_num)?;
        writeln!(f, "Hardware Version: {}", self.hw_ver)?;
        writeln!(f, "Software Version: {}", self.sw_ver)?;
        writeln!(f, "Uptime: {}:{:02}:{:02}", ut_hrs, ut_mins % 60, ut_secs)
    }
}

// take a byte measurement and convert to the smallest possible unit
pub fn format_bytes(bytes: u128) -> String {
    if calc_kilos(bytes) < 1.0 {
        format!("{}", bytes)
    }
    else if calc_megs(bytes) < 1.0 {
        format!("{:.2} KB", calc_kilos(bytes))
    }
    else if calc_gigs(bytes) < 1.0 {
        format!("{:.2} MB", calc_megs(bytes))
    }
    else {
        format!("{:.2} GB", calc_gigs(bytes))
    }
}

fn calc_kilos(bytes: u128) -> f64 { bytes as f64 / 1024f64 }
fn calc_megs(bytes: u128) -> f64 { bytes as f64 / 1048576f64 }
fn calc_gigs(bytes: u128) -> f64 { bytes as f64 / 1073741824f64 }