use super::*;
use reqwest::Url;
use rand::{Rng, thread_rng};
use std::{
    time::SystemTime,
    collections::HashMap
};
use sha2::{Sha256, Digest};

pub struct AuthToken {
    raw_token: String,
    sid_cookie: String,
    lsid_cookie: String,
    create_time: u64
}

impl AuthToken {
    // create a new token using a username and password
    pub fn get(client: &Client, username: &str, password: &str) -> Result<AuthToken, super::ClientError> {
        let nonce_response = match client.get("http://192.168.12.1/login_web_app.cgi?nonce").send() {
            Ok(response) => response.text().unwrap(),
            Err(why) => return Err(super::ClientError::GatewayUnreachable(why.to_string()))
        };

        let nonce_json = json::parse(&nonce_response).unwrap();

        // get the username and password hash
        let user_pass_hash = sha_256_hash(username, &password.to_lowercase());

        // hash the username+password hash and the nonce key together
        let nonce_key = nonce_json["nonce"].to_string();
        let passwd_nonce_hash = encode_as_url(sha_256_hash(&user_pass_hash, &nonce_key));
        
        // hash the username and nonce key together
        let username_nonce_hash = encode_as_url(sha_256_hash(username, &nonce_key));

        // get the random key number hash
        let rkey = nonce_json["randomKey"].to_string();
        let rkey_hash = encode_as_url(sha_256_hash(&rkey, &nonce_key));

        // generate the crypto keys
        let mut rand = thread_rng();
        let mut enckey1: Vec<u8> = Vec::new();
        let mut enckey2: Vec<u8> = Vec::new();

        for _ in 0..16 {
            enckey1.push(rand.gen());
            enckey2.push(rand.gen());
        }

        let mut request = HashMap::new();
        request.insert("userhash", username_nonce_hash);
        request.insert("RandomKeyhash", rkey_hash);
        request.insert("response", passwd_nonce_hash);
        request.insert("nonce", encode_as_url(nonce_key));
        request.insert("enckey", encode_as_url(base64::encode(enckey1)));
        request.insert("enciv", encode_as_url(base64::encode(enckey2)));

        let response = match client.post("http://192.168.12.1/login_web_app.cgi").form(&request).send() {
            Ok(r) => r,
            Err(why) => return Err(super::ClientError::GatewayUnreachable(why.to_string()))
        };

        let mut sid_cookie: String = String::new();
        let mut lsid_cookie: String = String::new();
        for cookie in response.cookies() {
            match cookie.name() {
                "sid" => sid_cookie = cookie.value().to_string(),
                "lsid" => lsid_cookie = cookie.value().to_string(),
                _ => { }
            }
        }
        let response_json = match json::parse(&response.text().unwrap()){
            Ok(json) => json,
            Err(_) => return Err(super::ClientError::LoginFailed)
        };

        Ok(AuthToken {
            raw_token: response_json["token"].to_string(),
            sid_cookie,
            lsid_cookie,
            create_time: SystemTime::now().duration_since(SystemTime::UNIX_EPOCH).unwrap().as_secs()
        })
    }

    // is this token more than 15 minutes old?
    pub fn is_expired(&self) -> bool {
        let cur_time = SystemTime::now().duration_since(SystemTime::UNIX_EPOCH).unwrap().as_secs();
        cur_time >= self.create_time + 900
    }

    // add the cookies of this token to the client's cookie store
    pub fn put_cookies_in_jar(&self, jar: &Arc<Jar>) {
        let sid = format!("sid={}; Domain=192.168.12.1", self.sid_cookie);
        let lsid = format!("lsid={}; Domain=192.168.12.1", self.lsid_cookie);
        let url = Url::parse("http://192.168.12.1").unwrap();
        jar.add_cookie_str(&sid, &url);
        jar.add_cookie_str(&lsid, &url);
    }

    // invalidates this authentication token
    // this is the equivalent of logging out of the web interface of the gateway
    pub fn invalidate(&self, client: &Client) -> Result<(), super::ClientError> {
        if let Err(why) = client.post("http://192.168.12.1/login_web_app.cgi?out").form(&self.get_data()).send() {
            return Err(super::ClientError::GatewayUnreachable(why.to_string()));
        }
        Ok(())
    }

    // get the session cookie needed to make requests
    pub fn get_data(&self) -> HashMap<&'static str, String> {
        let mut data = HashMap::new();
        data.insert("csrf_token", self.raw_token.clone());
        data
    }
}

fn sha_256_hash(val1: &str, val2: &str) -> String {
    let mut hasher = Sha256::new();
    let string_to_hash = format!("{val1}:{val2}");
    hasher.update(&string_to_hash);
    let output = hasher.finalize();
    base64::encode(&output[..])
}

fn encode_as_url(b64: String) -> String {
    let mut out = String::new();
    for c in b64.chars() {
        match c {
            '+' => out.push('-'),
            '/' => out.push('_'),
            '=' => out.push('.'),
            _ => out.push(c),
        }
    }
    out
}