use std::{env, io, process};
use crate::client::*;
use crate::{
    hosts::HostList,
    config::{Config, GlobalConfig},
    stats::*
};
use colored::Colorize;

const RATING: [&'static str; 5] = [
    "Poor",
    "Weak",
    "Good",
    "Very Good",
    "Excellent"
];

pub fn parse_args() {
    // get the command line arguments of the program
    let args: Vec<String> = env::args().collect();
    // create the client object and allow the user to enter the admin password if it is incorrect 
    let mut client = match TrashcanClient::new() {
        Ok(c) => c,
        Err(_) => {
            let mut passwd = String::new();
            let mut config = GlobalConfig::load();
            println!("WARNING! Your password will be visible!!");
            println!("Gateway admin password: ");
            io::stdin().read_line(&mut passwd).unwrap();
            passwd.retain(|c| c !='\n');
            config.password = passwd;
            match TrashcanClient::new() {
                Ok(c) => c,
                Err(_) => {
                    let msg = "Unable to Log into the gateway. Is your admin password correct?".bright_red();
                    eprintln!("{}", msg);
                    return;
                }
            }
        }
    };
    
    if args.len() <= 1 {
        show_stats(&get_stats(&client));
        return;
    }

    match args[1].as_str() {
        "-h" => {
            println!("tmocli help:");
            println!("Use tmocli -h to show this again.");
            println!("Use tmocli -v to show current version.");
            println!("Subcommands:");
            println!("  devices: list all devices connected to gateway");
            println!("  device <name, IP, or MAC address>: find device connected to gateway by name, IP address or MAC address");
            println!("  stats: show the gateway's stats");
            println!("  data-usage, data: show the gateway's data usage");
            println!("  networks, net: show the gateway's active networks");
            println!("  connection, con: show the gateway's cell connection status");
            println!("  info: show general info about the gateway");
            println!("  reboot: reboot the gateway");
        },
        "-v" => {
            println!("T-Mobile Home Internet Tools\nVersion {}", env!("CARGO_PKG_VERSION"));
        },
        "devices" => show_devices(client),
        "device" => {
            let arg = match args.get(2) {
                Some(a) => a.to_string(),
                None => {
                    println!("Please specify a device name, IP address, or MAC address!");
                    return;
                }
            };
            show_device(client, arg);
        },
        "stats" => show_stats(&get_stats(&client)),
        "data-usage" | "data" => show_data(&get_stats(&client)),
        "networks" | "net" => show_networks(&get_stats(&client)),
        "connection" | "con" => show_con_stat(&get_stats(&client)),
        "info" => show_info(client),
        "reboot" => {
            ctrlc::set_handler(|| {
                println!("Reboot Cancelled!");
                process::exit(0);
            }).expect("Error setting ^C handler!");

            let mut ans = String::new();
            println!("Your network will be down for about 5 minutes. Reboot gateway?(N/y)");
            io::stdin().read_line(&mut ans).unwrap();
            ans.retain(|c| c !='\n');
            if ans.as_str() != "y" {
                println!("Reboot Cancelled!");
                return;
            }
            println!("Rebooting gateway...");
            if let Err(why) = client.reboot_gateway_no_log() {
                let msg = format!("Error rebooting gateway: {}", why);
                eprintln!("{}", msg);
                return;
            };
        },
        _ => eprintln!("Invalid option!")
    }
}

fn show_info(client: TrashcanClient) {
    match client.get_info() {
        Ok(i) => {
            println!("Gateway Info:");
            println!("{}", i);
        }
        Err(why) => {
            let msg = format!("Error getting gateway info: {}", why);
                eprintln!("{}", msg);
                return;
        }
    }
}

fn show_stats(stats: &GatewayStats) {
    show_data(stats);
    show_networks(stats);
    show_con_stat(stats);
}

fn show_networks(stats: &GatewayStats) {
    println!("Active Networks:\n");
    for network in &stats.networks {
        if network.enabled {
            let download_fmt = format!("Down: {}", format_bytes(network.download)).bright_magenta();
            let upload_fmt = format!("Up: {}", format_bytes(network.upload)).bright_cyan();
            println!("{}", network.name);
            println!("ID: {}", network.id);
            println!("{} {}\n", download_fmt, upload_fmt);
        }
    }
}

fn show_data(stats: &GatewayStats) {
    println!("Data Usage:");
    let download_fmt = format!("Download: {}", format_bytes(stats.download)).bright_magenta();
    let upload_fmt = format!("Upload: {}", format_bytes(stats.upload)).bright_cyan();
    let total_fmt = format!("Total: {}", format_bytes(stats.download + stats.upload));
    println!("{} {} {}\n", download_fmt, upload_fmt, total_fmt);
}

fn show_con_stat(stats: &GatewayStats) {
    println!("Connection Status:\n");
    println!("5G:");
    println!("Band: {}", stats.info_5g.band);
    println!("Strength: {} dBm ({})\n", stats.info_5g.strength, RATING[(stats.info_5g.bars - 1) as usize]);
    println!("LTE (4G):");
    println!("Band: {}", stats.info_4g.band);
    println!("Strength: {} dBm ({})", stats.info_4g.strength, RATING[(stats.info_4g.bars - 1) as usize]);
}

fn show_device(client: TrashcanClient, arg: String) {
    let hosts = get_devices(&client);
    for host in &hosts.hosts {
        if host.get_name() == arg || host.get_ip() == arg || host.get_mac_addr() == arg{
            println!("{}", host.get_name().bright_magenta());
            println!("{}", host.get_ip());
            if !host.is_gateway() {
                println!("{}", host.get_mac_addr());
            }
        }
    }
}

fn show_devices(client: TrashcanClient) {
    let hosts = get_devices(&client);
    for host in &hosts.hosts {
        println!("{}", host.get_name().bright_magenta());
        println!("{}", host.get_ip());
        if !host.is_gateway() {
            println!("{}", host.get_mac_addr());
        }
        println!();
    }
    print!("\x08");
}

fn get_devices(client: &TrashcanClient) -> HostList {
    match client.get_hosts() {
        Ok(h) => h,
        Err(why) => {
            let msg = format!("Error: Unable to get devices connected to gateway! {}", why).bright_red();
            eprintln!("{}", msg);
            process::exit(255);
        }
    }
}

// get the gateway's stats
fn get_stats(client: &TrashcanClient) -> GatewayStats {
        match client.get_all_stats() {
            Ok(s) => s,
            Err(why) => {
                let msg = format!("Error: Unable to get gateway stats! {}", why).bright_red();
                eprintln!("{}", msg);
                process::exit(255);
            }
        }
}