use compact_str::CompactStr;
use isocountry::CountryCode;
use isolanguage_1::LanguageCode;
use serde::{Deserialize, Deserializer};
use serde_with::serde_as;
use time::Date;

time::serde::format_description!(date, Date, "[year]-[month]-[day]");

fn country_code<'de, D: Deserializer<'de>>(de: D) -> Result<Option<CountryCode>, D::Error> {
	use serde::de::{Error, Unexpected};
	match Option::<&str>::deserialize(de)? {
		None => Ok(None),
		Some("") => Ok(None),
		Some(s) if s.len() == 2 => match CountryCode::for_alpha2_caseless(s) {
			Ok(v) => Ok(Some(v)),
			Err(_) => Err(D::Error::invalid_value(Unexpected::Str(s), &"any 2-letter ISO 3166-1 country code"))
		},
		Some(s) => Err(D::Error::invalid_value(Unexpected::Str(s), &"any 2-letter ISO 3166-1 country code"))
	}
}

mod imdb_id {
	use serde::{Deserialize, Deserializer};
	use serde::de::{Error, Unexpected};

	fn from_str<'de, D: Deserializer<'de>>(s: &str) -> Result<u32, D::Error> {
		if(s.len() < 3 || &s[..2] != "tt") {
			return Err(D::Error::invalid_value(Unexpected::Str(s), &"a signed integer prefixed by \"tt\""));
		}
		match s[2..].parse() {
			Ok(v) => Ok(v),
			Err(_) => Err(D::Error::invalid_value(Unexpected::Str(s), &"a signed integer prefixed by \"tt\""))
		}
	}

	pub(crate) fn deserialize<'de, D: Deserializer<'de>>(de: D) -> Result<u32, D::Error> {
		let s = <&str>::deserialize(de)?;
		from_str::<D>(s)
	}

	pub(super) mod option {
		use super::*;
		pub(crate) fn deserialize<'de, D: Deserializer<'de>>(de: D) -> Result<Option<u32>, D::Error> {
			match Option::<&str>::deserialize(de)? {
				Some(s) => Ok(Some(from_str::<D>(s)?)),
				None => Ok(None)
			}
		}
	}
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct Genre {
	pub id: u16,
	pub name: CompactStr,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct Results<T> {
	pub results: Vec<T>,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct Video {
	pub id: CompactStr,
	pub iso_639_1: LanguageCode,
	pub key: CompactStr,
	pub name: CompactStr,
	pub site: CompactStr,
	pub size: u16,
	#[serde(rename = "type")]
	pub video_type: CompactStr,
}

#[serde_as]
#[derive(Debug, PartialEq, Deserialize)]
pub struct Cast {
	pub id: u32,
	pub cast_id: u32,
	#[serde_as(as = "serde_with::hex::Hex")]
	pub credit_id: [u8; 12],
	pub character: CompactStr,
	pub gender: Option<u8>,
	pub name: CompactStr,
	pub profile_path: Option<String>,
	pub order: u8,
}

#[serde_as]
#[derive(Debug, PartialEq, Deserialize)]
pub struct TVCast {
	pub id: u32,
	#[serde_as(as = "serde_with::hex::Hex")]
	pub credit_id: [u8; 12],
	pub character: CompactStr,
	pub gender: Option<u8>,
	pub name: CompactStr,
	pub profile_path: Option<String>,
	pub order: u32,
}

#[serde_as]
#[derive(Debug, PartialEq, Deserialize)]
pub struct TVCreator {
	pub id: u32,
	#[serde_as(as = "serde_with::hex::Hex")]
	pub credit_id: [u8; 12],
	pub name: CompactStr,
	pub gender: Option<u8>,
	pub profile_path: Option<String>,
}

#[serde_as]
#[derive(Debug, PartialEq, Deserialize)]
pub struct Crew {
	#[serde_as(as = "serde_with::hex::Hex")]
	pub credit_id: [u8; 12],
	pub department: CompactStr,
	pub gender: Option<u8>,
	pub id: u32,
	pub job: CompactStr,
	pub name: CompactStr,
	pub profile_path: Option<String>,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct Credits {
	pub cast: Vec<Cast>,
	pub crew: Vec<Crew>,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct TVCredits {
	pub cast: Vec<TVCast>,
	pub crew: Vec<Crew>,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct LastEpisode {
	#[serde(with = "date")]
	pub air_date: Date,
	pub episode_number: u32,
	pub id: u32,
	pub name: CompactStr,
	pub overview: String,
	pub production_code: Option<String>,
	pub season_number: u32,
	pub still_path: Option<String>,
	pub vote_average: f64,
	pub vote_count: u64,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct ProductionCompany {
	pub id: u32,
	pub logo_path: Option<String>,
	pub name: CompactStr,
	#[serde(deserialize_with = "country_code")]
	pub origin_country: Option<CountryCode>,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct Network {
	pub id: u32,
	pub logo_path: Option<String>,
	pub name: CompactStr,
	#[serde(deserialize_with = "country_code")]
	pub origin_country: Option<CountryCode>,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct Season {
	#[serde(with = "date::option")]
	pub air_date: Option<Date>,
	pub episode_count: u32,
	pub id: u32,
	pub name: CompactStr,
	pub overview: String,
	pub poster_path: Option<String>,
	pub season_number: u32,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct Movie {
	pub id: u32,
	#[serde(with = "imdb_id")]
	pub imdb_id: u32,
	pub title: CompactStr,
	pub tagline: String,
	pub original_title: CompactStr,
	pub original_language: LanguageCode,
	pub overview: Option<String>,
	#[serde(with = "date")]
	pub release_date: Date,
	pub runtime: u32,
	pub homepage: Option<String>,
	pub genres: Vec<Genre>,
	pub poster_path: Option<String>,
	pub backdrop_path: Option<String>,
	pub popularity: f64,
	pub budget: u64,
	pub adult: bool,
	pub videos: Option<Results<Video>>,
	pub credits: Option<Credits>,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct TV {
	pub id: u32,
	pub backdrop_path: Option<String>,
	pub created_by: Vec<TVCreator>,
	pub episode_run_time: Vec<u16>,
	#[serde(with = "date")]
	pub first_air_date: Date,
	pub genres: Vec<Genre>,
	pub homepage: Option<String>,
	pub in_production: bool,
	pub languages: Vec<LanguageCode>,
	#[serde(with = "date")]
	pub last_air_date: Date,
	pub last_episode_to_air: Option<LastEpisode>,
	pub name: CompactStr,
	pub networks: Vec<Network>,
	pub number_of_episodes: u32,
	pub number_of_seasons: u32,
	pub origin_country: Vec<CountryCode>,
	pub original_language: LanguageCode,
	pub original_name: CompactStr,
	pub overview: String,
	pub popularity: f64,
	pub poster_path: Option<CompactStr>,
	pub production_companies: Vec<ProductionCompany>,
	pub seasons: Vec<Season>,
	pub status: CompactStr,
	pub r#type: CompactStr,
	pub vote_average: f64,
	pub vote_count: u64,
	pub videos: Option<Results<Video>>,
	pub credits: Option<TVCredits>,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct SearchMovie {
	pub id: u32,
	pub title: CompactStr,
	pub original_title: CompactStr,
	pub original_language: LanguageCode,
	pub overview: Option<String>,
	#[serde(with = "date")]
	pub release_date: Date,
	pub genre_ids: Vec<u16>,
	pub poster_path: Option<String>,
	pub backdrop_path: Option<String>,
	pub popularity: f64,
	pub adult: bool,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct SearchTV {
	pub id: u32,
	pub name: CompactStr,
	pub original_name: CompactStr,
	pub original_language: LanguageCode,
	pub overview: Option<String>,
	#[serde(with = "date")]
	pub first_air_date: Date,
	pub genre_ids: Vec<u16>,
	pub poster_path: Option<String>,
	pub backdrop_path: Option<String>,
	pub popularity: f64,
	pub vote_average: f32,
	pub vote_count: u32,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct FindMovie {
	pub id: u32,
	pub title: CompactStr,
	pub original_title: CompactStr,
	pub original_language: LanguageCode,
	pub overview: Option<String>,
	#[serde(with = "date")]
	pub release_date: Date,
	pub genre_ids: Vec<u16>,
	pub poster_path: Option<String>,
	pub backdrop_path: Option<String>,
	pub adult: bool,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct FindTV {
	pub id: u32,
	pub name: CompactStr,
	pub original_name: CompactStr,
	pub original_language: LanguageCode,
	pub overview: Option<String>,
	#[serde(with = "date")]
	pub first_air_date: Date,
	pub genre_ids: Vec<u16>,
	pub poster_path: Option<String>,
	pub backdrop_path: Option<String>,
	pub popularity: f64,
	pub vote_average: f32,
	pub vote_count: u32,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct MovieSearchResult {
	pub page: u8,
	pub total_results: u8,
	pub total_pages: u8,
	pub results: Vec<SearchMovie>,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct TVSearchResult {
	pub page: u8,
	pub total_results: u8,
	pub total_pages: u8,
	pub results: Vec<SearchTV>,
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct FindResult {
	pub movie_results: Vec<FindMovie>,
	pub tv_results: Vec<FindTV>
}

#[derive(Debug, PartialEq, Deserialize)]
pub struct TVExternalIds {
	pub id: u32,
	#[serde(with = "imdb_id::option")]
	pub imdb_id: Option<u32>,
	pub freebase_mid: Option<CompactStr>,
	pub freebase_id: Option<CompactStr>,
	pub tvdb_id: Option<u32>,
	pub tvrage_id: Option<u32>,
	pub facebook_id: Option<CompactStr>,
	pub instagram_id: Option<CompactStr>,
	pub twitter_id: Option<CompactStr>
}
