use itertools::Itertools;
use reqwest::Error;
use serde::de::DeserializeOwned;
use smartstring::alias::String as SmartString;

mod model;
use model::FindResult;
pub use model::{Movie, MovieSearchResult, TV, TVSearchResult};

#[cfg(test)]
mod integration_tests;

const BASE_URL: &str = "https://api.themoviedb.org/3";

#[derive(Debug, Clone)]
pub struct Client {
	http: reqwest::Client,
	api_key: String,
	language: String
}

impl Client {
	pub fn new(api_key: String) -> Self {
		Self::with_language(api_key, "en".into())
	}

	pub fn with_language(api_key: String, language: String) -> Self {
		Self{
			http: reqwest::Client::new(),
			api_key,
			language
		}
	}
	
	async fn get<T: DeserializeOwned>(&self, path: &str, args: &[(&'static str, SmartString)]) -> Result<T, Error> {
		let url = format!(
			"{}{}?api_key={}&language={}&{}",
			BASE_URL,
			path,
			self.api_key,
			self.language,
			args.iter().map(|(k, v)| SmartString::from(*k) + "=" + v).join("&")
		);
		dbg!(&url);
		self.http.get(url).send().await?.json().await
	}

	pub async fn movie_search(&self, title: &str, year: Option<u64>) -> Result<MovieSearchResult, Error> {
		let mut args = Vec::with_capacity(3);
		args.push(("query", title.into()));
		if let Some(year) = year {
			args.push(("year", year.to_string().into()));
		}
		args.push(("append_to_response", "images".into()));
		self.get("/search/movie", &args).await
	}

	pub async fn movie_by_id(&self, id: u64, include_videos: bool, include_credits: bool) -> Result<Movie, Error> {
		let args = match (include_videos, include_credits) {
			(false, false) => None,
			(true, false) => Some(("append_to_response", "videos".into())),
			(false, true) => Some(("append_to_response", "credits".into())),
			(true, true) => Some(("append_to_response", "videos,credits".into()))
		};
		self.get(&format!("/movie/{}", id), args.as_ref().map(core::slice::from_ref).unwrap_or_default()).await
	}

	pub async fn movie_by_imdb_id(&self, id: u64) -> Result<Movie, Error> {
		let result: FindResult = self.get(&format!("/find/tt{:07}", id), &[
			("external_source", "imdb_id".into()),
			("append_to_response", "images".into())
		]).await?;
		self.movie_by_id(result.movie_results[0].id, false, false).await
	}

	pub async fn tv_search(&self, title: &str, year: Option<u64>) -> Result<TVSearchResult, Error> {
		let mut args = Vec::with_capacity(3);
		args.push(("query", title.into()));
		if let Some(year) = year {
			args.push(("year", year.to_string().into()));
		}
		args.push(("append_to_response", "images".into()));
		self.get("/search/tv", &args).await
	}

	pub async fn tv_by_id(&self, id: u64, include_videos: bool, include_credits: bool) -> Result<TV, Error> {
		let args = match (include_videos, include_credits) {
			(false, false) => None,
			(true, false) => Some(("append_to_response", "videos".into())),
			(false, true) => Some(("append_to_response", "credits".into())),
			(true, true) => Some(("append_to_response", "videos,credits".into()))
		};
		self.get(&format!("/tv/{}", id), args.as_ref().map(core::slice::from_ref).unwrap_or_default()).await
	}

	pub async fn tv_by_imdb_id(&self, id: u64) -> Result<TV, Error> {
		let result: FindResult = self.get(&format!("/find/tt{}", id), &[
			("external_source", "imdb_id".into()),
			("append_to_response", "images".into())
		]).await?;
		self.tv_by_id(result.tv_results[0].id, false, false).await
	}

	pub async fn tv_by_tvdb_id(&self, id: u32) -> Result<TV, Error> {
		let result: FindResult = self.get(&format!("/find/{}", id), &[
			("external_source", "tvdb_id".into()),
			("append_to_response", "images".into())
		]).await?;
		self.tv_by_id(result.tv_results[0].id, false, false).await
	}
}

