use crate::Client;

const API_KEY: &str = env!("TMDB_API_KEY");

fn client() -> Client {
	Client::new(API_KEY.into())
}

#[tokio::test]
async fn fetch_movie() {
	let client = client();
	let movie = client.movie_by_id(157336, false, false).await.unwrap();

	assert_eq!("Interstellar", movie.original_title);
}

#[tokio::test]
async fn fetch_movie_languages() {
	let client = Client::with_language(API_KEY.into(), "en".into());
	let movie = client.movie_by_id(2277, false, false).await.unwrap();
	assert_eq!("Bicentennial Man", movie.title);

	let client = Client::with_language(API_KEY.into(), "de".into());
	let movie = client.movie_by_id(2277, false, false).await.unwrap();
	assert_eq!("Der 200 Jahre Mann", movie.title);

	let client = Client::with_language(API_KEY.into(), "es".into());
	let movie = client.movie_by_id(2277, false, false).await.unwrap();
	assert_eq!("El hombre bicentenario", movie.title);
}

#[tokio::test]
async fn fetch_movie_append_to_response() {
	let client = client();
	let movie = client.movie_by_id(2277, true, true).await.unwrap();
	assert_eq!(true, movie.videos.is_some());
	assert_eq!(true, movie.credits.is_some());
}

#[tokio::test]
async fn search_movie() {
	let client = client();
	let page = client.movie_search("Bicentennial Man", Some(1999)).await.unwrap();

	assert_eq!(1, page.total_results);
	assert_eq!("Bicentennial Man", page.results[0].title);
}

#[tokio::test]
async fn find_movie_by_imdb_id() {
	let client = client();
	let movie = client.movie_by_imdb_id(816692).await.unwrap();

	assert_eq!("Interstellar", movie.title);
}

#[tokio::test]
async fn fetch_searched_movie() {
	let client = client();
	let page = client.movie_search("Bicentennial Man", Some(1999)).await.unwrap();
	let movie = client.movie_by_id(page.results[0].id, false, false).await.unwrap();

	assert_eq!(2277, movie.id);
}

#[tokio::test]
async fn fetch_tv() {
	let client = client();
	let series = client.tv_by_id(2316, false, false).await.unwrap();

	assert_eq!("The Office", series.original_name);
}

#[tokio::test]
async fn fetch_tv_languages() {
	// TODO:  The Flash might have been a poor choice, since its name appears
	//    to be the same in all languages.  Let's find a show that differs from
	//    English to German to Spanish.
	let client = Client::with_language(API_KEY.into(), "en".into());
	let series = client.tv_by_id(60735, false, false).await.unwrap();
	assert_eq!("The Flash", series.name);

	let client = Client::with_language(API_KEY.into(), "de".into());
	let series = client.tv_by_id(60735, false, false).await.unwrap();
	assert_eq!("The Flash", series.name);

	let client = Client::with_language(API_KEY.into(), "es".into());
	let series = client.tv_by_id(60735, false, false).await.unwrap();
	assert_eq!("The Flash", series.name);
}

#[tokio::test]
async fn fetch_tv_append_to_response() {
	let client = client();
	let series = client.tv_by_id(2316, true, true).await.unwrap();

	assert_eq!(true, series.videos.is_some());
	assert_eq!(true, series.credits.is_some());
}

#[tokio::test]
async fn search_tv() {
	let client = client();
	let page = client.tv_search("The Simpsons", Some(1989)).await.unwrap();

	assert_eq!(1, page.total_results);
	assert_eq!("The Simpsons", page.results[0].name);
}

#[tokio::test]
async fn find_tv_by_imdb_id() {
	let client = client();
	let series = client.tv_by_imdb_id(1520211).await.unwrap();

	assert_eq!("The Walking Dead", series.name);
}

#[tokio::test]
async fn find_tv_by_tvdb_id() {
	let client = client();
	let series = client.tv_by_tvdb_id(94571).await.unwrap();

	assert_eq!("Community", series.name);
}

#[tokio::test]
async fn fetch_searched_tv() {
	let client = client();
	let page = client.tv_search("Futurama", None).await.unwrap();
	let series = client.tv_by_id(page.results[0].id, false, false).await.unwrap();

	assert_eq!(615, series.id);
}

