//! A crate to implement the [TLDMP](https://files.almaember.com/tld-messages.txt) protocol.
//! Example usage:
//! ```rust
//! 
//! fn main() {
//!     let mut output: Vec<u8> = Vec::new();
//! 
//!     tld_msg::send(&mut output, 1, &mut "Hey!".as_bytes()).unwrap();
//!     let mut input = &output[..];
//!     let msg = tld_msg::receive(&mut input).unwrap();
//! 
//!     println!("{:#?}", output);
//!     println!("{:#?}\n{}", msg, String::from_utf8(msg.content.clone()).unwrap());
//! }
//! ```

use std::io::{Error, Read, Write};

/// Struct for storing a message
#[derive(Debug)]
pub struct Message {
    /// The type field of the message, decoded
    /// from Little-Endian (if neccessary)
    pub mtype: u8,
    /// The content of the message as raw bytes
    pub content: Vec<u8>,
}

/// Used to send a message through a Write implementation
/// 
/// Example:
/// ```rust
/// tld_msg::send(&mut output, 1, &mut "Hey!".as_bytes()).unwrap();
/// ```
pub fn send(wr: &mut impl Write, mtype: u8, msg: &[u8]) -> Result<(), Error> {
    wr.write(&mtype.to_le_bytes())?;

    let len64: u64 = msg.len().try_into().unwrap();
    wr.write_all(&len64.to_le_bytes())?;
    wr.write_all(msg)?;

    Result::Ok(())
}

/// Used to read a message from a Read implementation
/// 
/// Example:
/// ```rust
/// let msg = tld_msg::receive(&mut input).unwrap();
/// ```
pub fn receive(rd: &mut impl Read) -> Result<Message, Error> {
    let mtype = {
        let mut buf: [u8; 1] = [0];
        rd.read_exact(&mut buf)?;
        buf[0]
    };

    let length: usize = {
        let mut buf = [0u8; 8];
        rd.read_exact(&mut buf)?;

        u64::from_le_bytes(buf).try_into().unwrap()
    };

    let mut buf = vec![0u8; length];
    rd.read_exact(&mut buf)?;

    Ok(Message {
        mtype,
        content: buf,
    })
}
