use std::pin::Pin;
use std::future::Future;
use std::collections::HashMap;

use tokio::sync::mpsc;

use super::{ClientMessage, RouterMessage, Path};

/// A message received by a `Service` implementation.
#[derive(Debug)]
pub struct ServiceMessage<'a> {
    /// Path to the service (id of the service)
    pub path: &'a Path,
    /// The raw payload (minus the path)
    pub payload: &'a [u8],
    /// The client sender. Use this to send
    /// messages back to the client
    pub sender: mpsc::Sender<ClientMessage>,
    /// Client connection id
    pub id: usize,
}

// -----------------------------------------------------------------------------
//     - Service -
// -----------------------------------------------------------------------------
/// Service trait.
/// Has to be implemented for anything used with the router.
///
/// ```
/// use std::pin::Pin;
/// use std::future::Future;
/// use tinyroute::server::{Router, ServiceMessage, Service};
///
/// struct MyService { }
///
/// impl Service for MyService {
///     fn handle<'a>(&'a mut self, msg: ServiceMessage<'a>) -> Pin<Box<dyn Future<Output=()> + Send + 'a>> {
///         let async_nonsense = async move {};
///         Box::pin(async_nonsense)
///     }
/// }
///
/// let mut router = Router::builder();
/// router.register("my-service", MyService {});
/// ```
pub trait Service {
    /// Handle incoming service messages
    fn handle<'a>(&'a mut self, _msg: ServiceMessage<'a>) -> Pin<Box<dyn Future<Output=()> + Send + 'a>> {
        let async_block = async move {};
        Box::pin(async_block)
    }

    /// A client unsubscribe from this service
    fn unsubscribe(&mut self, _id: usize) {}

    /// When a service is registered with the router it receives
    /// a `Sender<RouterMessage>`. This is useful
    /// to register serivces at runtime etc.
    fn registered(&mut self, _tx: mpsc::Sender<RouterMessage>) {
    }

    /// List of routes
    fn routes(&self, _routes: &HashMap<Path, Box<dyn Service + Send>>) {
    }
}

