use std::collections::HashMap;
use std::fmt;

use tokio::sync::mpsc;
use log::info;

use super::{Server, Service, ServiceMessage, ClientMessage};

/// A path to a [`Service`]
#[derive(Debug, Eq, PartialEq, Hash, Clone)]
pub struct Path(Vec<u8>);

impl From<Vec<u8>> for Path {
    fn from(inner: Vec<u8>) -> Self {
        Self(inner)
    }
}

impl From<&[u8]> for Path {
    fn from(inner: &[u8]) -> Self {
        Self(inner.to_vec())
    }
}

impl From<&str> for Path {
    fn from(inner: &str) -> Self {
        Self(inner.as_bytes().to_vec())
    }
}

impl From<String> for Path {
    fn from(inner: String) -> Self {
        Self(inner.into_bytes())
    }
}

impl fmt::Display for Path {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match std::str::from_utf8(&self.0) {
            Ok(s) => write!(f, "{}", s),
            Err(_) => write!(f, "[path not valid utf8]"),
        }
    }
}

macro_rules! some_or_cont {
    ($e:expr) => {
        match $e {
            Some(v) => v,
            None => continue,
        }
    }
}

/// A message sent to the router.
pub enum RouterMessage {
    /// Request a hash map with the routes
    Routes(Path),
    /// Register a service.
    /// Useful to register services dynamically.
    RegisterService(Path, Box<dyn Service + Send>),
    /// When a client is disconnected it sends a `ClientDisconnected` message
    /// to the router for each [`Service`], if and only if the [`Service`] has sent a
    /// [`crate::server::ClientMessage::TrackService`] message to the client.
    ClientDisconnected(Path, usize),
    /// Sent when a service should be removed
    RemoveService(Path),
    /// A message received from a connected client
    Payload { 
        /// Message bytes
        payload: Vec<u8>, 
        /// Connection sender
        sender: mpsc::Sender<ClientMessage>, 
        /// Connection id
        id: usize 
    },
}

/// ```
/// # use tinyroute::server::{Service, Router, TcpServer};
/// # async fn run(service: impl Service + Send + 'static) {
///     let mut router = Router::builder();
///     router
///         .register("my-service", service)
///         .timeout(30)
///         .channel_size(1);
///
///     let tcp_server = TcpServer::bind("127.0.0.1:12345").await.unwrap();
///     router.serve(tcp_server).await;
/// # }
/// ```
pub struct RouterBuilder {
    routes: HashMap<Path, Box<dyn Service + Send>>,
    tx: mpsc::Sender<RouterMessage>,
    rx: mpsc::Receiver<RouterMessage>,
    timeout: Option<u64>,
    channel_size: usize
}

impl RouterBuilder {
    /// Get a copy of the routers `tokio::mpsc::Sender<RouterMessage>`.
    pub fn sender(&self) -> mpsc::Sender<RouterMessage> {
        self.tx.clone()
    }

    /// Register a [`Service`] at a given [`Path`]
    pub fn register(&mut self, path: impl Into<Path>, mut service: impl Service + Send + 'static) -> &mut Self {
        service.registered(self.tx.clone());
        self.routes.insert(path.into(), Box::new(service));
        self
    }

    /// Set the connetion timeout.
    /// This is `None` by default, meaning a connection will never timeout.
    pub fn timeout(&mut self, timeout: u64) -> &mut Self {
        self.timeout = Some(timeout);
        self
    }

    /// Set the number of messages that can go into a connection's channel
    /// before it's full.
    ///
    /// See
    /// [https://docs.rs/tokio/1.9.0/tokio/sync/broadcast/fn.channel.html](https://docs.rs/tokio/1.9.0/tokio/sync/broadcast/fn.channel.html)
    /// for more information about the capacity.
    pub fn channel_size(&mut self, size: usize) -> &mut Self {
        self.channel_size = size;
        self
    }

    /// Starts the server.
    pub async fn serve(self, server: impl Server + Send + 'static) -> Option<()> {
        let RouterBuilder { routes, tx, mut rx, timeout, channel_size } = self;

        let mut router = Router { routes, tx: tx.clone() };

        tokio::spawn(super::serve(server, tx, timeout, channel_size));

        loop {
            let msg = rx.recv().await?;

            match msg {
                RouterMessage::Routes(path) => {
                    let service = match router.routes.get(&path) {
                        Some(s) => s,
                        None => continue,
                    };

                    let routes = &router.routes;
                    service.routes(routes);
                },
                RouterMessage::RemoveService(path) => {
                    let _ = router.routes.remove(&path);
                    info!("number of routes after removal: {}", router.routes.len());
                }
                RouterMessage::RegisterService(path, mut service) => {
                    service.registered(router.tx.clone());
                    router.routes.insert(path, service);
                }
                RouterMessage::ClientDisconnected(path, id) => {
                    let route = some_or_cont!(router.routes.get_mut(&path));
                    route.unsubscribe(id);
                }
                RouterMessage::Payload { payload, sender, id } => {
                    let path = payload
                        .iter()
                        .cloned()
                        .take_while(|b| (*b as char) != '|') // <- doc this
                        .collect::<Vec<u8>>();
                    let path = Path(path);

                    let route = some_or_cont!(router.routes.get_mut(&path));

                    // We need to offset the message 
                    // to start where the id ends, + the 
                    // command byte
                    let offset = 1 + path.0.len();
                    if offset >= payload.len() {
                        continue;
                    }

                    // Finally we can construct the message payload
                    route.handle(ServiceMessage { payload: &payload[offset..], path: &path, sender, id }).await;
                }
            }

        }
    }
}

/// Route messages to services.
///
/// ```
/// use tinyroute::server::{Service, Router};
/// # struct AService;
/// # impl Service for AService {}
///
/// let mut router = Router::builder();
/// router.register("path-to-service", AService {});
/// ```
// -----------------------------------------------------------------------------
//     - Router -
// -----------------------------------------------------------------------------
pub struct Router {
    routes: HashMap<Path, Box<dyn Service + Send>>,
    tx: mpsc::Sender<RouterMessage>,
}

impl Router {
    /// Create a new instance of a router
    pub fn builder() -> RouterBuilder {
        let (tx, rx) = mpsc::channel(100);
        RouterBuilder {
            routes: HashMap::new(),
            tx,
            rx, 
            channel_size: 100,
            timeout: None,
        }
    }
}
