//! ```
//! use std::pin::Pin;
//! use std::future::Future;
//! use tinyroute::server::{
//!     Service, Router, TcpServer, ServiceMessage, ClientMessage 
//! };
//!
//! struct MyService {
//!     message_counter: usize,
//! }
//! 
//! impl Service for MyService {
//!     fn handle<'a>(&'a mut self, msg: ServiceMessage<'a>) -> Pin<Box<dyn Future<Output=()> + Send + 'a>> {
//!         let async_block = async move {
//!             let s = std::str::from_utf8(msg.payload).unwrap();
//!             eprintln!("message: {}", s);
//!             msg.sender.send(ClientMessage::Payload("hello".into()));
//!             self.message_counter += 1;
//!         };
//!         Box::pin(async_block)
//!     }
//!
//!     fn unsubscribe(&mut self, id: usize) {
//!         eprintln!("Client {} is no longer subscribed", id);
//!     }
//! }
//!
//! #[tokio::main]
//! async fn main() {
//!     let first_service = MyService { message_counter: 0 };
//!     let mut router = Router::builder();
//!     router.register("my-service", first_service);
//!     let tcp_server = TcpServer::bind("127.0.0.1:12345").await.unwrap();
//!     # let _ = async move {
//!     router.serve(tcp_server).await;
//!     # };
//! }
//! ```
//!
//! For more information about the `Service` trait see [`crate::server::Service`].
//!
//! For more information about the `Client` messages see [`crate::server::ClientMessage`].
//!
use std::future::Future;
use std::pin::Pin;
use std::time::Duration;

use log::{error, info};
use tokio::io::{AsyncRead, AsyncWrite, AsyncWriteExt};
use tokio::sync::mpsc;

use crate::errors::{TinyRouteError, Result};
use crate::Frame;

mod tcp;
mod service;
mod router;

pub use router::{RouterMessage, Router, Path};
pub use tcp::TcpServer;
pub use service::{ServiceMessage, Service};

/// Type alias for a [`RouterMessage`] sender
pub type RouterSender = mpsc::Sender<RouterMessage>;
/// Type alias for a [`ClientMessage`] sender
pub type ClientSender = mpsc::Sender<ClientMessage>;

#[cfg(target_os = "linux")]
mod uds;
#[cfg(target_os = "linux")]
pub use uds::UdsServer;

/// Message sent back to the client (the connection)
#[derive(Debug, PartialEq)]
pub enum ClientMessage {
    /// Tell the client connection to close, with 
    /// an optional message.
    ///
    /// ```
    /// use tinyroute::server::ClientMessage;
    /// # fn run() {
    ///     let text = "connection closed, all is well";
    ///     let msg = ClientMessage::Quit(Some(text.into()));
    /// }
    /// ```
    Quit(Option<Vec<u8>>),
    /// The client subscribed to a service.
    ///
    /// For every service that sends this message to a client, the client will 
    /// send a `RouterMessage::ClientDisconnected(path, id)` to the service
    /// once the connection is closed.
    ///
    /// This is useful for cleaning up services that tracks the clients.
    TrackService(Path),
    /// A message from a service
    Payload(bytes::Bytes),
}

/// Some kind of listener
pub trait Server: Sync {
    /// The reading half of the connection
    type Reader: AsyncRead + Unpin + Send + 'static;
    /// The writing half of the connection
    type Writer: AsyncWrite + Unpin + Send + 'static;

    // Accepts &self as arg
    // Returns a pinned boxed future, where
    // * any reference has to live for at least as long as &self,
    // * and it has to be valid to send this across thread boundaries
    //
    // We need the `Send` part because tokio::spawn might put this on another thread.
    // We need the life time because the thing we return can not hold a reference to
    // anything on &self that might be dropped before self.
    /// Accept incoming connections
    fn accept( &mut self,) -> TraitFuture<'_, Self::Reader, Self::Writer>;
}

/// Because writing this entire trait malarkey is messy!
pub type TraitFuture<'a, T, U> = Pin<Box<dyn Future<Output = Result<(T, U)>> + Send + 'a>>;

pub(crate) async fn serve(
    mut server: impl Server,
    router_tx: mpsc::Sender<RouterMessage>,
    timeout: Option<u64>,
    channel_size: usize,
) {
    let mut next_id = 0;
    loop {
        if let Ok((reader, writer)) = server.accept().await {
            let (tx, rx) = mpsc::channel(channel_size);
            tokio::spawn(use_reader(
                reader,
                next_id,
                tx.clone(),
                router_tx.clone(),
                timeout
            ));
            tokio::spawn(use_writer(writer, next_id, rx, router_tx.clone()));
            next_id += 1;
        }
    }
}

async fn use_writer(
    mut writer: impl AsyncWrite + Unpin,
    id: usize,
    mut rx: mpsc::Receiver<ClientMessage>,
    router_tx: mpsc::Sender<RouterMessage>
) {
    let mut services = Vec::new();

    loop {
        match rx.recv().await {
            Some(ClientMessage::Quit(None)) => break,
            Some(ClientMessage::Quit(Some(payload))) => {
                let msg = Frame::frame_message(&payload);
                let _ = writer.write_all(&msg.0).await;
                break
            }
            Some(ClientMessage::TrackService(path)) => services.push(path),
            Some(ClientMessage::Payload(payload)) => { 
                let message = Frame::frame_message(&payload);
                match writer.write_all(&message.0).await {
                    Ok(_) => {}
                    Err(e) => {
                        error!("Failed to write message: {:?}", e);
                        break
                    }
                }
            }
            None => {
                error!("receiver closed");
                break
            }
        }
    }

    for s in services {
        info!("dropping: {:?}", s);
        drop(router_tx.send(RouterMessage::ClientDisconnected(s, id)).await);
    }

    info!("Closing (writer)");
}

async fn use_reader(
    mut reader: impl AsyncRead + Unpin,
    id: usize,
    writer_tx: mpsc::Sender<ClientMessage>,
    router_tx: mpsc::Sender<RouterMessage>,
    timeout: Option<u64>,
) {
    let mut frame = Frame::empty();

    loop {
        let read = frame.async_read(&mut reader);

        let timeout_fut = match timeout {
            Some(t) => tokio::time::sleep(Duration::from_secs(t)),
            None => tokio::time::sleep(Duration::from_secs(999)),
        };

        tokio::pin!(timeout_fut);

        tokio::select! {
            res = read => {
                match res {
                    Ok(0) => break,
                    Ok(_) => {
                        match frame.try_msg() {
                            Ok(None) => continue,
                            Ok(Some(payload)) => {
                                let msg = RouterMessage::Payload {
                                    payload, 
                                    sender: writer_tx.clone(),
                                    id
                                };
                                let _ = router_tx.send(msg).await;
                            },
                            Err(TinyRouteError::MalformedHeader) => {
                                drop(writer_tx.send(ClientMessage::Quit(Some("Connection closed: malformed header\n".as_bytes().to_owned()))).await);
                                break
                            },
                            Err(_) => unreachable!(),
                        }
                    }
                    Err(e) => {
                        error!("Connection closed: {:?}", e);
                        break;
                    }
                }
            }

            // Timeout
            _ = &mut timeout_fut, if timeout.is_some() => {
                eprintln!("{:?}", "if you can read this it means I've failed in life");
                break
            }
        }
    }

    info!("Closing (reader)");
    let _ = writer_tx.send(ClientMessage::Quit(None)).await;
}
