// A (very particular kind of) chat example.
// To connect use something like `netcat` or `telnet`.
//
// netcat 127.0.0.1 5000
//
// To join a group:
//
// ```<group name>|/join```
//
// To leave a group:
//
// ```<group name>|/leave```
//
// To send a message to everyone in the group:
// 
// ```<group name>|<message>```

use std::future::Future;
use std::pin::Pin;
use std::collections::HashMap;

use bytes::Bytes;
use tokio::sync::mpsc;
use log::{error, info};
use pretty_env_logger;

use tinyroute::server::{TcpServer, ClientMessage, ServiceMessage, Service, Router, RouterMessage};


struct Group {
    members: HashMap<usize, mpsc::Sender<ClientMessage>>,
    router_tx: mpsc::Sender<RouterMessage>,
}

impl Group {
    fn new(router_tx: mpsc::Sender<RouterMessage>) -> Self {
        Self {
            members: HashMap::new(),
            router_tx,
        }
    }
}

impl Service for Group {
    fn unsubscribe(&mut self, id: usize) {
        info!("removing connection id: {}", id);
        self.members.remove(&id);
    }

    fn handle<'a>(&'a mut self, msg: ServiceMessage<'a>) -> Pin<Box<dyn Future<Output=()> + Send + 'a>> {
        let async_block = async move {

            let (cmd, payload) = if *&msg.payload[0] == b'/' {
                let cmd = msg
                .payload
                .iter()
                .cloned()
                .take_while(|b| !(*b as char).is_whitespace())
                .collect::<Vec<_>>();
                let offset = cmd.len();
                info!("Command: {:?}", std::str::from_utf8(&cmd));
                (cmd, &msg.payload[offset..])
            } else { (vec![], msg.payload) };

            match &cmd as &[u8] {
                b"/join" => {
                    if !self.members.iter().any(|(&id, _)| id == msg.id) {
                        let _ = msg.sender.send(ClientMessage::TrackService(msg.path.clone())).await;
                        self.members.insert(msg.id, msg.sender);
                    }
                    info!("join: {}", msg.path);
                }
                b"/members" => {
                    let output = format!("members: {}\n", self.members.len());
                    let bytes = Bytes::from(output);
                    let _ = msg.sender.send(ClientMessage::Payload(bytes)).await;
                }
                b"/leave" => {
                    self.unsubscribe(msg.id);
                    if self.members.len() == 0 {
                        let _ = self.router_tx.send(RouterMessage::RemoveService(msg.path.clone())).await;
                    }
                }
                _ => {
                    info!("payload size: {}", payload.len());

                    let bytes = Bytes::from(payload.to_vec());
                    for (_, m) in self.members.iter() {
                        if let Err(e) = m.send(ClientMessage::Payload(bytes.clone())).await {
                            error!("Failed to send message: {:?}", e);
                        }
                    }
                }
            }
        };

        Box::pin(async_block)
    }
}

struct GroupsService {
    router_tx: mpsc::Sender<RouterMessage>,
}

impl Service for GroupsService {
    fn handle<'a>(&'a mut self, msg: ServiceMessage<'a>) -> Pin<Box<dyn Future<Output=()> + Send + 'a>> {
        let fut = async move {
            if msg.payload.starts_with(b"/create ") {
                let group_name = &msg.payload[8..];
                info!("group creation: {:?}", std::str::from_utf8(group_name));
                let _ = self.router_tx.send(
                    RouterMessage::RegisterService(
                        group_name.into(), Box::new(Group::new(self.router_tx.clone()))
                    )
                ).await;
            }

            if msg.payload.starts_with(b"/groups ") {
            }

            eprintln!("msg: {:?}", std::str::from_utf8(msg.payload).unwrap());
        };
        Box::pin(fut)
    }
}

#[tokio::main]
async fn main() {
    pretty_env_logger::init();

    let message = r#"
-----------------------------------------------------------------------------
 A (very particular kind of) chat example.
 To connect use something like `netcat` or `telnet`.
-----------------------------------------------------------------------------

$ netcat 127.0.0.1 5000

To join a group:
<group name>|/join

To leave a group:
<group name>|/leave

To send a message to everyone in the group:
<group name>|<message>
"#;

    println!("{}", message);

    let mut router_builder = Router::builder();
    let groups_service = GroupsService { router_tx: router_builder.sender() };
    router_builder.register(b"!".to_vec(), groups_service);
    let tcp_server = TcpServer::bind("127.0.0.1:5000").await.unwrap();
    router_builder.serve(tcp_server).await;
}
