extern crate tinymt;

use tinymt::tinymt64::*;
use tinymt::TinyMT64;

/// https://github.com/MersenneTwister-Lab/TinyMT/blob/master/tinymt/check64.c
#[test]
fn test_cases_by_author() {
  // tinymt64 0xfa051f40 0xffd0fff4 0x58d02ffeffbfffbc seed = 1
  let mut tinymt = TinyMT64::new([0, 0], 0xfa051f40, 0xffd0fff4, 0x58d02ffeffbfffbc);
  let seed = 1;
  tinymt64_init(&mut tinymt, seed);

  test_generate_uint64(&mut tinymt);

  let mut tinymt = TinyMT64::new([0, 0], 0xfa051f40, 0xffd0fff4, 0x58d02ffeffbfffbc);

  // init_by_array {1}
  let seed_array: [u64; 1] = [1];
  tinymt64_init_by_array(&mut tinymt, &seed_array);

  test_generate_double(&mut tinymt);
  test_generate_double12(&mut tinymt);
  test_generate_double_oc(&mut tinymt);
  test_generate_double_oo(&mut tinymt);
}

/// 64-bit unsigned integers r, where 0 <= r < 2^64
fn test_generate_uint64(tinymt: &mut TinyMT64) {
  let expected = [
    [15503804787016557143, 17280942441431881838, 2177846447079362065],
    [10087979609567186558, 8925138365609588954, 13030236470185662861],
    [4821755207395923002, 11414418928600017220, 18168456707151075513],
    [1749899882787913913, 2383809859898491614, 4819668342796295952],
    [11996915412652201592, 11312565842793520524, 995000466268691999],
    [6363016470553061398, 7460106683467501926, 981478760989475592],
    [11852898451934348777, 5976355772385089998, 16662491692959689977],
    [4997134580858653476, 11142084553658001518, 12405136656253403414],
    [10700258834832712655, 13440132573874649640, 15190104899818839732],
    [14179849157427519166, 10328306841423370385, 9266343271776906817],
  ];
  for i in 0..10 {
    for j in 0..3 {
      assert_eq!(expected[i][j], tinymt64_generate_uint64(tinymt));
    }
  }
}

/// double numbers r, where 0.0 <= r < 1.0
fn test_generate_double(tinymt: &mut TinyMT64) {
  let expected = [
    [0.1255671232295209, 0.8182624006077499, 0.30822110203281683, 0.8255918229908551],
    [0.2555517877036223, 0.8826415608914364, 0.21152361685256493, 0.31910695814713397],
    [0.8731938455315581, 0.7563442179617009, 0.9626867074958626, 0.12749130989590807],
    [0.6701740931329137, 0.09321519841996262, 0.4752574502959318, 0.4260656296146129],
    [0.8342582498085203, 0.2790339713179786, 0.5149478695059739, 0.036174029320189205],
    [0.8897244223874985, 0.2517822059311857, 0.38744954297723777, 0.08612504349103156],
    [0.7653487864751842, 0.5710556579917725, 0.3222522710904576, 0.02403074050158116],
    [0.08852883324737215, 0.9634180500801006, 0.45325188731530386, 0.9794639730992276],
    [0.5959355146678814, 0.5271370962042583, 0.4095847647067309, 0.004006194821549625],
    [0.20031375905173576, 0.4542823501864448, 0.713777124755221, 0.8079152811762041],
    [0.12756425652308678, 0.9863577232725115, 0.5371878217659848, 0.23544098948491765],
    [0.7000063536628773, 0.8580450245315596, 0.13056501891409378, 0.17396590319565852],
  ];
  for i in 0..12 {
    for j in 0..4 {
      assert_eq!(expected[i][j], tinymt64_generate_double(tinymt));
    }
  }
}

/// double numbers r, where 1.0 <= r < 2.0
fn test_generate_double12(tinymt: &mut TinyMT64) {
  let expected = [
    [1.437679237017648, 1.239536785901373, 1.140298949383057, 1.776408301859232],
    [1.152013609994736, 1.791233026870471, 1.212111221146196, 1.829985488180836],
    [1.081512125943717, 1.363201836673650, 1.417933283495315, 1.814826826183523],
    [1.969922345279833, 1.053208264502199, 1.741205427976973, 1.837349090361589],
    [1.406622310957582, 1.510698317360325, 1.829965206684917, 1.859153888163104],
    [1.759271635641173, 1.824888617384633, 1.237637472413003, 1.367109059723164],
    [1.976389381199251, 1.989991431835970, 1.044503045383735, 1.769751873156083],
    [1.859046544898330, 1.218170930629463, 1.308291384260259, 1.694324347868131],
    [1.458264916022492, 1.128833025697983, 1.205547655611532, 1.909188848740936],
    [1.562083063485982, 1.333329220907858, 1.665680038183793, 1.001161742007127],
    [1.667546697634258, 1.296057871298311, 1.461095987795535, 1.459580681054313],
    [1.556093077958318, 1.916051394545249, 1.267046730316243, 1.147033584842960],
  ];
  for i in 0..12 {
    for j in 0..4 {
      let actual = tinymt64_generate_double12(tinymt);
      assert_eq!(format!("{:.15}", expected[i][j]), format!("{:.15}", actual));
    }
  }
}

/// double numbers r, where 0.0 < r <= 1.0
fn test_generate_double_oc(tinymt: &mut TinyMT64) {
  let expected = [
    [0.231189305675805, 0.800078680337062, 0.839012626265816, 0.439830027924101],
    [0.287094637016178, 0.588065859945908, 0.979935435454641, 0.153150392249384],
    [0.730008781559804, 0.811897304025850, 0.213940001686070, 0.803418052576349],
    [0.872859727831960, 0.620437548528132, 0.978150212926246, 0.101173021131322],
    [0.910550586203282, 0.225948191636215, 0.374316722183833, 0.305990832583114],
    [0.349868211955804, 0.617675089001072, 0.977990275060935, 0.453879799720974],
    [0.894692817941832, 0.247166853705171, 0.639918430646982, 0.187528433375713],
    [0.098449225468909, 0.630303237374302, 0.080830809996716, 0.578706622599148],
    [0.734493648961771, 0.082578413837076, 0.287252902600609, 0.148892860351310],
    [0.032585155152626, 0.974545363240716, 0.115296495734384, 0.779122282518212],
    [0.217374280466345, 0.109080092606533, 0.926570354675966, 0.740322917071261],
    [0.751484666825263, 0.287404891102534, 0.652825666825707, 0.715421981731271],
  ];
  for i in 0..12 {
    for j in 0..4 {
      let actual = tinymt64_generate_double_oc(tinymt);
      assert_eq!(format!("{:.15}", expected[i][j]), format!("{:.15}", actual));
    }
  }
}

/// double numbers r, where 0.0 <= r < 1.0
fn test_generate_double_oo(tinymt: &mut TinyMT64) {
  let expected = [
    [0.777528512172794, 0.427164705471249, 0.646272649811224, 0.544192399276788],
    [0.055887665337890, 0.341700406526459, 0.711593276934271, 0.834779506059920],
    [0.300433789431423, 0.551665825454914, 0.408923581859956, 0.087661367648074],
    [0.778332952183721, 0.987791992333503, 0.372489630499866, 0.814160794049290],
    [0.729786510846933, 0.759103094898038, 0.126259258963777, 0.126254138650957],
    [0.959804243067140, 0.297513369134440, 0.080559577448319, 0.759676389010768],
    [0.299289034712620, 0.930880432389953, 0.453686608702271, 0.051097960521366],
    [0.190779343357538, 0.668131291883840, 0.568395939590751, 0.051087427533098],
    [0.531001654494900, 0.863620810621520, 0.713312386789292, 0.079647214591833],
    [0.503865458699875, 0.649652561747019, 0.651303365948799, 0.423447062299514],
    [0.383204776309473, 0.325776002558352, 0.917474566663556, 0.612145094607316],
    [0.785176513330525, 0.046217567933767, 0.445326162565375, 0.426594677789800],
  ];
  for i in 0..12 {
    for j in 0..4 {
      let actual = tinymt64_generate_double_oo(tinymt);
      assert_eq!(format!("{:.15}", expected[i][j]), format!("{:.15}", actual));
    }
  }
}
