#![doc = include_str!("../README.md")]
#![doc(html_root_url = "https://docs.rs/tinylog/1.0.0")]
#![forbid(unsafe_code)]
#![warn(missing_docs)]
#![allow(clippy::tabs_in_doc_comments)]

use log::{Level, LevelFilter, Log, Metadata, Record};
use termcolor::{BufferedStandardStream, Color, ColorChoice, ColorSpec, WriteColor};
use std::{
	env::{self, VarError},
	io::{self, Write},
	sync::Mutex,
};

#[cfg(feature = "humantime")]
use std::time::SystemTime;

#[cfg(debug_assertions)]
const DEFAULT_LEVEL: LevelFilter = LevelFilter::Debug;
#[cfg(not(debug_assertions))]
const DEFAULT_LEVEL: LevelFilter = LevelFilter::Info;

struct Logger {
	level: LevelFilter,
	stdout: Mutex<BufferedStandardStream>,
}

/// Initialize the logger.
///
/// Any logs that occur before this are ignored.
#[allow(clippy::needless_doctest_main)]
pub fn init() {
	let level = match env::var("RUST_LOG") {
		Err(err) => match err {
			VarError::NotPresent => DEFAULT_LEVEL,
			VarError::NotUnicode(_) => panic!("RUST_LOG must be valid unicode"),
		},

		Ok(val) => match &*val {
			"error" => LevelFilter::Error,
			"warn" => LevelFilter::Warn,
			"info" => LevelFilter::Info,
			"debug" => LevelFilter::Debug,
			"trace" => LevelFilter::Trace,
			_ => panic!(
				"RUST_LOG must be one of: error, warn, info, debug, trace, off\ngot: {}",
				val
			),
		},
	};

	let color_choice = if env::var("FORCE_COLOR").is_ok() {
		ColorChoice::Always
	} else if atty::is(atty::Stream::Stdout) {
		ColorChoice::Auto
	} else {
		ColorChoice::Never
	};

	log::set_max_level(level);
	log::set_boxed_logger(Box::new(Logger {
		level,
		stdout: Mutex::new(BufferedStandardStream::stdout(color_choice)),
	}))
	.expect("error setting logger");
}

impl Log for Logger {
	fn enabled(&self, meta: &Metadata) -> bool {
		self.level >= meta.level()
	}

	fn log(&self, record: &Record) {
		if self.enabled(record.metadata()) {
			self.real_log(record).expect("log error");
		}
	}

	fn flush(&self) {}
}

impl Logger {
	fn real_log(&self, record: &Record) -> io::Result<()> {
		#[cfg(feature = "humantime")]
		let time = humantime::format_rfc3339_seconds(SystemTime::now());

		let (color, should_dim) = match record.level() {
			Level::Error => (Color::Red, false),
			Level::Warn => (Color::Yellow, false),
			Level::Info => (Color::Green, false),
			Level::Debug => (Color::Blue, true),
			Level::Trace => (Color::Cyan, true),
		};

		let mut color = {
			let mut spec = ColorSpec::new();
			spec.set_fg(Some(color))
				.set_bold(true)
				.set_dimmed(should_dim);
			spec
		};

		let mut stdout = self.stdout.lock().expect("stream poisoned");

		#[cfg(feature = "humantime")]
		stdout.set_color(&*ColorSpec::new().set_dimmed(true))?;
		#[cfg(feature = "humantime")]
		write!(stdout, "{} ", time)?;

		stdout.set_color(&color)?;
		write!(stdout, "{:>5} ", record.level())?;
		color.set_bold(false).set_intense(false);
		stdout.set_color(&color)?;
		write!(stdout, "({}) ", record.target())?;
		color.set_fg(None);
		stdout.set_color(&color)?;
		writeln!(stdout, "{}", record.args())?;
		stdout.reset()?;
		stdout.flush()
	}
}
