// Copyright 2020 The Tink-Rust Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
////////////////////////////////////////////////////////////////////////////////

//! Factory methods for [`tink_core::Signer`] instances.

use std::sync::Arc;
use tink_core::{utils::wrap_err, TinkError};

/// Return a [`tink_core::Signer`] primitive from the given keyset handle.
pub fn new_signer(h: &tink_core::keyset::Handle) -> Result<Box<dyn tink_core::Signer>, TinkError> {
    new_signer_with_key_manager(h, None)
}

/// Return a [`tink_core::Signer`] primitive from the given keyset handle and custom key manager.
fn new_signer_with_key_manager(
    h: &tink_core::keyset::Handle,
    km: Option<Arc<dyn tink_core::registry::KeyManager>>,
) -> Result<Box<dyn tink_core::Signer>, TinkError> {
    let ps = h
        .primitives_with_key_manager(km)
        .map_err(|e| wrap_err("signer::factory: cannot obtain primitive set", e))?;

    let ret = WrappedSigner::new(ps)?;
    Ok(Box::new(ret))
}

/// A [`tink_core::Signer`] implementation that uses the underlying primitive set for signing.
#[derive(Clone)]
struct WrappedSigner {
    ps: tink_core::primitiveset::TypedPrimitiveSet<Box<dyn tink_core::Signer>>,
}

impl WrappedSigner {
    fn new(ps: tink_core::primitiveset::PrimitiveSet) -> Result<WrappedSigner, TinkError> {
        let primary = match &ps.primary {
            None => return Err("signer::factory: no primary primitive".into()),
            Some(p) => p,
        };
        match primary.primitive {
            tink_core::Primitive::Signer(_) => {}
            _ => return Err("signer::factory: not a Signer primitive".into()),
        };
        for (_, primitives) in ps.entries.iter() {
            for p in primitives {
                match p.primitive {
                    tink_core::Primitive::Signer(_) => {}
                    _ => return Err("signer::factory: not a Signer primitive".into()),
                };
            }
        }
        // The `.into()` call is only safe because we've just checked that all entries have
        // the right type of primitive
        Ok(WrappedSigner { ps: ps.into() })
    }
}

impl tink_core::Signer for WrappedSigner {
    /// Sign the given data and returns the signature concatenated with the identifier of the
    /// primary primitive.
    fn sign(&self, data: &[u8]) -> Result<Vec<u8>, TinkError> {
        let primary = match &self.ps.primary {
            Some(p) => p,
            None => return Err("signer::factory: no primary primitive".into()),
        };

        let signature = if primary.prefix_type == tink_proto::OutputPrefixType::Legacy {
            let mut signed_data_copy = Vec::with_capacity(data.len() + 1);
            signed_data_copy.extend_from_slice(data);
            signed_data_copy.push(tink_core::cryptofmt::LEGACY_START_BYTE);
            primary.primitive.sign(&signed_data_copy)?
        } else {
            primary.primitive.sign(data)?
        };

        let mut ret = Vec::with_capacity(primary.prefix.len() + signature.len());
        ret.extend_from_slice(&primary.prefix);
        ret.extend_from_slice(&signature);
        Ok(ret)
    }
}
