use alloc::{
    collections::BinaryHeap,
    sync::{Arc, Weak},
};
use core::{cmp::Ordering, ops::Add, task::Poll};

use futures::{task::AtomicWaker, Future};
use num::Zero;

use crate::mutex::Mutex;

#[derive(Clone, Debug)]
pub struct Timewrap<T: Ord + Zero + Clone> {
    data: Arc<Mutex<TimewrapData<T>>>,
}

#[derive(Debug)]
struct TimewrapData<T: Ord + Zero + Clone> {
    current: T,
    wakers: BinaryHeap<Arc<TimeTask<T>>>,
}

#[derive(Debug)]
struct TimeTask<T: Ord + Zero + Clone> {
    time: T,
    data: Weak<Mutex<TimewrapData<T>>>,
    waker: AtomicWaker,
}

impl<T: Ord + Zero + Clone> PartialEq for TimeTask<T> {
    fn eq(&self, other: &Self) -> bool {
        self.time == other.time
    }
}

impl<T: Ord + Zero + Clone> Eq for TimeTask<T> {}

impl<T: Ord + Zero + Clone> PartialOrd for TimeTask<T> {
    fn partial_cmp(&self, other: &Self) -> Option<core::cmp::Ordering> {
        self.time.partial_cmp(&other.time).map(Ordering::reverse)
    }
}

impl<T: Ord + Zero + Clone> Ord for TimeTask<T> {
    fn cmp(&self, other: &Self) -> core::cmp::Ordering {
        self.time.cmp(&other.time).reverse()
    }
}

pub trait Moment {
    type Time: Ord + Zero + Clone;
    fn at_time(self, current: Self::Time) -> Self::Time;
}

impl<T: Ord + Zero + Clone> Moment for T {
    type Time = Self;
    fn at_time(self, _: Self::Time) -> Self::Time {
        self
    }
}

#[derive(PartialEq, Eq, PartialOrd, Ord, Clone, Copy, Debug, Default)]
pub struct FromCurrent<T: Ord + Zero + Clone>(T);

impl<T: Ord + Zero + Clone> Moment for FromCurrent<T> {
    type Time = T;
    fn at_time(self, current: Self::Time) -> Self::Time {
        self.0 + current
    }
}

pub struct Current;

impl<T: Ord + Zero + Clone> Add<T> for Current {
    type Output = FromCurrent<T>;
    fn add(self, rhs: T) -> Self::Output {
        FromCurrent(rhs)
    }
}

impl<T: Ord + Zero + Clone> TimewrapData<T> {
    fn at(slf: &Arc<Mutex<Self>>, time: impl Moment<Time = T>) -> At<T> {
        let mut data = slf.lock();
        let tt = Arc::new(TimeTask {
            time: time.at_time(data.current.clone()),
            data: Arc::downgrade(&slf),
            waker: AtomicWaker::new(),
        });
        data.wakers.push(tt.clone());
        At { inner: tt }
    }
}

impl<T: Ord + Zero + Clone> Timewrap<T> {
    pub fn new() -> Self {
        Self::new_with(T::zero())
    }

    pub fn new_with(current: T) -> Self {
        Self {
            data: Arc::new(Mutex::new(TimewrapData {
                current,
                wakers: BinaryHeap::new(),
            })),
        }
    }

    pub fn at(&self, time: impl Moment<Time = T>) -> At<T> {
        TimewrapData::at(&self.data, time)
    }

    pub fn delay(&self, time: T) -> At<T> {
        TimewrapData::at(&self.data, Current + time)
    }

    pub fn forward(&self, time: impl Moment<Time = T>) {
        let mut data = self.data.lock();
        data.current = time.at_time(data.current.clone());
        while let Some(task) = data.wakers.peek() {
            if task.time > data.current {
                break;
            }
            task.waker.wake();
            data.wakers.pop();
        }
    }
}

#[derive(Debug)]
pub struct At<T: Ord + Zero + Clone> {
    inner: Arc<TimeTask<T>>,
}

impl<T: Ord + Zero + Clone> Clone for At<T> {
    fn clone(&self) -> Self {
        match self.inner.data.upgrade() {
            Some(data) if data.lock().current < self.inner.time => {
                TimewrapData::at(&data, self.inner.time.clone())
            }
            _ => Self {
                inner: self.inner.clone(),
            },
        }
    }
}

impl<T: Ord + Zero + Clone> Future for At<T> {
    type Output = ();
    fn poll(
        self: core::pin::Pin<&mut Self>,
        cx: &mut core::task::Context<'_>,
    ) -> core::task::Poll<Self::Output> {
        match self.inner.data.upgrade() {
            Some(data) if data.lock().current < self.inner.time => {
                self.inner.waker.register(cx.waker());
                Poll::Pending
            }
            _ => Poll::Ready(()),
        }
    }
}
