use alloc::sync::Arc;
use core::{fmt::Debug, pin::Pin};
use futures::Future;
use num::Zero;
use tokio::{
    runtime::Runtime,
    task::{yield_now, JoinHandle},
};

use crate::{
    timewrap::{At, Moment},
    Timewrap,
};

#[derive(Clone)]
pub struct TimewrapRuntime<T: Ord + Zero + Clone> {
    runtime: Arc<Runtime>,
    timewrap: Timewrap<T>,
}

impl<T: Ord + Zero + Clone> TimewrapRuntime<T> {
    pub fn new() -> Self {
        Self::new_with(T::zero())
    }

    pub fn new_with(current: T) -> Self {
        Self {
            runtime: Arc::new(
                tokio::runtime::Builder::new_current_thread()
                    .build()
                    .unwrap(),
            ),
            timewrap: Timewrap::new_with(current),
        }
    }

    pub fn spawn<F>(&self, f: F) -> Spawn<F::Output>
    where
        F: Future + Send + 'static,
        F::Output: Send,
    {
        Spawn(self.runtime.spawn(f))
    }

    pub fn at(&self, time: impl Moment<Time = T>) -> At<T> {
        self.timewrap.at(time)
    }

    pub fn delay(&self, time: T) -> At<T> {
        self.timewrap.delay(time)
    }

    pub fn drive(&self, time: impl Moment<Time = T>) {
        self.timewrap.forward(time);
        self.runtime.block_on(yield_now());
    }
}

pub struct Spawn<T>(JoinHandle<T>);

impl<T> Future for Spawn<T> {
    type Output = T;
    fn poll(
        self: core::pin::Pin<&mut Self>,
        cx: &mut core::task::Context<'_>,
    ) -> core::task::Poll<Self::Output> {
        unsafe { Pin::new_unchecked(&mut self.get_unchecked_mut().0) }
            .poll(cx)
            .map(|r| match r {
                Ok(r) => r,
                Err(r) => panic!("{}", r),
            })
    }
}

impl<T> Debug for Spawn<T>
where
    T: Debug,
{
    fn fmt(&self, f: &mut core::fmt::Formatter<'_>) -> core::fmt::Result {
        Debug::fmt(&self.0, f)
    }
}
