use core::{
    borrow::{Borrow, BorrowMut},
    fmt::{Debug, Display},
};
use core::{
    cell::UnsafeCell,
    hint,
    ops::{Deref, DerefMut},
    sync::atomic::{AtomicBool, Ordering},
};

pub struct Mutex<T> {
    lock: AtomicBool,
    item: UnsafeCell<T>,
}

pub struct MutexGuard<'a, T> {
    target: &'a Mutex<T>,
}

impl<T> Mutex<T> {
    pub fn new(value: T) -> Self {
        Self {
            item: UnsafeCell::new(value),
            lock: AtomicBool::new(false),
        }
    }

    pub fn try_lock(&self) -> Option<MutexGuard<T>> {
        if self.lock.swap(true, Ordering::Acquire) {
            None
        } else {
            Some(MutexGuard { target: self })
        }
    }

    pub fn lock(&self) -> MutexGuard<T> {
        loop {
            if let Some(guard) = self.try_lock() {
                return guard;
            } else {
                hint::spin_loop();
            }
        }
    }
}

impl<T> Deref for MutexGuard<'_, T> {
    type Target = T;
    fn deref(&self) -> &Self::Target {
        unsafe { &*self.target.item.get() }
    }
}

impl<T> DerefMut for MutexGuard<'_, T> {
    fn deref_mut(&mut self) -> &mut Self::Target {
        unsafe { &mut *self.target.item.get() }
    }
}

impl<T> Drop for MutexGuard<'_, T> {
    fn drop(&mut self) {
        let _prev = self.target.lock.swap(false, Ordering::Release);
        debug_assert!(_prev);
    }
}

impl<T> AsRef<T> for MutexGuard<'_, T> {
    fn as_ref(&self) -> &T {
        &*self
    }
}

impl<T> AsMut<T> for MutexGuard<'_, T> {
    fn as_mut(&mut self) -> &mut T {
        &mut *self
    }
}

impl<T> Borrow<T> for MutexGuard<'_, T> {
    fn borrow(&self) -> &T {
        &*self
    }
}

impl<T> BorrowMut<T> for MutexGuard<'_, T> {
    fn borrow_mut(&mut self) -> &mut T {
        &mut *self
    }
}

unsafe impl<T> Send for Mutex<T> where T: Send {}
unsafe impl<T> Sync for Mutex<T> where T: Send {}

unsafe impl<T> Send for MutexGuard<'_, T> where T: Send {}
unsafe impl<T> Sync for MutexGuard<'_, T> where T: Send + Sync {}

impl<T: Clone> Clone for Mutex<T> {
    fn clone(&self) -> Self {
        Self::new(T::clone(&*self.lock()))
    }
}

impl<T: Debug> Debug for Mutex<T> {
    fn fmt(&self, f: &mut core::fmt::Formatter<'_>) -> core::fmt::Result {
        Debug::fmt(&*self.lock(), f)
    }
}

impl<T: Display> Display for Mutex<T> {
    fn fmt(&self, f: &mut core::fmt::Formatter<'_>) -> core::fmt::Result {
        Display::fmt(&*self.lock(), f)
    }
}

impl<T: PartialEq> PartialEq for Mutex<T> {
    fn eq(&self, other: &Self) -> bool {
        &*self.lock() == &*other.lock()
    }
}

impl<T: Eq> Eq for Mutex<T> {}

impl<T: PartialOrd> PartialOrd for Mutex<T> {
    fn partial_cmp(&self, other: &Self) -> Option<core::cmp::Ordering> {
        (&*self.lock()).partial_cmp(&*other.lock())
    }
}

impl<T: Ord> Ord for Mutex<T> {
    fn cmp(&self, other: &Self) -> core::cmp::Ordering {
        (&*self.lock()).cmp(&*other.lock())
    }
}
