#![allow(dead_code)]

use std::fs::{self, File, OpenOptions};
use std::io::Write;
use std::path::PathBuf;

use chrono::prelude::*;

#[derive(Debug)]
pub struct ActivityPaths {
    past: PathBuf,
    current: PathBuf,
}

#[derive(Debug, PartialEq)]
pub struct Activity {
    pub start: DateTime<Utc>,
    pub end: Option<DateTime<Utc>>,
}

#[derive(Debug, PartialEq)]
pub enum StartActivityError {
    CannotCreateFile,
    AlreadyExistingCurrentActivity,
}

#[derive(Debug, PartialEq)]
pub enum StopActivityError {
    NoExistingCurrentActivity,
}

impl Activity {
    pub fn start() -> Result<Activity, StartActivityError> {
        // check if there's already a current activity - cannot have more than one at same time
        let activity_paths = setup_files();

        match activity_paths.current.exists() {
            true => return Err(StartActivityError::AlreadyExistingCurrentActivity),
            false => match File::create(&activity_paths.current) {
                Ok(_) => (),
                Err(_) => return Err(StartActivityError::CannotCreateFile),
            },
        };

        let mut current_activity_file = OpenOptions::new()
            .write(true)
            .append(true)
            .open(activity_paths.current)
            .expect("unable to open file");

        let activity = Activity {
            start: Utc::now(),
            end: None,
        };
        writeln!(current_activity_file, "{:?}", activity.start).expect("unable to write");
        Ok(activity)
    }

    pub fn stop() -> Result<(), StopActivityError> {
        // check if there is a current activity
        let activity_paths = setup_files();

        let current_activity_file = match File::open(&activity_paths.current) {
            Ok(v) => v,
            Err(_) => return Err(StopActivityError::NoExistingCurrentActivity),
        };

        // TODO
        //  - create function to parse out this file into a Activity

        // delete the current_activity file
        fs::remove_file(activity_paths.current).unwrap();
        Ok(())
    }

    pub fn track(start: DateTime<Utc>, end: DateTime<Utc>) -> Result<Activity, String> {
        unimplemented!();
    }
}

/// Create dir in user's home dir and file for past activities
fn setup_files() -> ActivityPaths {
    let mut path = dirs::config_dir().unwrap();
    path.push("timestudy");

    // TODO: handle properly
    if !path.exists() {
        fs::create_dir(&path).unwrap();
    }

    let mut past_activities_path = path.clone();
    past_activities_path.push("past_activities");

    let mut current_activity_path = path;
    current_activity_path.push("current_activity");

    // TODO - handle properly
    if !past_activities_path.exists() {
        File::create(&past_activities_path).expect("Unable to create file");
    }
    ActivityPaths {
        past: past_activities_path,
        current: current_activity_path,
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn creating_activity_succeeds() {
        let a = Activity::start().unwrap();
        assert_eq!(a.end, None)
    }

    #[test]
    fn disallow_two_current_activities() {
        let _ = Activity::start();
        assert_eq!(
            Activity::start(),
            Err(StartActivityError::AlreadyExistingCurrentActivity)
        );
    }
}
