mod common;

use futures::StreamExt;
use timesource::error::Error;
use timesource::store::{CommitOrder, EventStore, EventStoreBuilder};
use timesource::subscription::SubscriptionBuilder;
use tokio::task::yield_now;
use uuid::Uuid;

use crate::common::data::{bootstrap_test, sample_events_uncommitted, TestData};
use crate::common::order::{OrderCommand, OrderEvent, OrderItem};

#[tokio::test]
async fn repository_should_reconstruct_state_after_appending_all_variants_of_events(
) -> Result<(), Box<dyn std::error::Error>> {
    let TestData {
        aggregate_id,
        mut repository,
        mut root,
        ..
    } = bootstrap_test(false).await;

    // append unit, tuple and struct enum variants
    root.handle(OrderCommand::Create).await?;
    root.handle(OrderCommand::AddItem {
        item: OrderItem {
            item_sku: "sku-123".into(),
            quantity: 1,
            price: 12,
        },
    })
    .await?;
    root.handle(OrderCommand::Empty("some reason".into()))
        .await?;

    repository.commit_orderly(&mut root).await?;

    let stored = repository.get(aggregate_id).await?;

    let (stored_state, _) = stored.state().to_owned().unwrap();
    let (expected_state, _) = root.state().unwrap();

    assert_eq!(
        expected_state.items, stored_state.items,
        "Stored items are not what's expected"
    );
    assert_eq!(
        expected_state.created_at, stored_state.created_at,
        "Order created_at timestamp doesn't match"
    );

    Ok(())
}

#[tokio::test]
async fn repository_should_err_if_not_found() {
    let TestData { repository, .. } = bootstrap_test(false).await;

    let output = repository.get(Uuid::new_v4()).await;

    match output {
        Ok(_) => unreachable!(),
        Err(Error::AggregateRootNotFound) => (),
        Err(_) => unreachable!(),
    }
}

#[tokio::test]
#[should_panic(expected = "Aggregate(AlreadyCreated)")]
async fn should_propagate_aggregate_err() {
    let TestData {
        pool,
        aggregate_type_name,
        repository,
        aggregate_id,
        ..
    } = bootstrap_test(false).await;

    let mut store = EventStoreBuilder::new(pool.clone())
        .build::<OrderEvent>(aggregate_type_name.into())
        .await
        .expect("store to be created");

    // Order aggregate should fail with invalid events
    store
        .commit(
            aggregate_id,
            CommitOrder::None,
            &[OrderEvent::Created.into(), OrderEvent::Created.into()],
        )
        .await
        .unwrap();

    repository.get(aggregate_id).await.unwrap();
}

#[tokio::test]
async fn subscription_should_fetch_prior_events_on_resume() {
    let TestData {
        aggregate_id,
        mut repository,
        mut root,
        subscription,
        ..
    } = bootstrap_test(false).await;

    //
    // Append event before resuming subscription
    //
    root.handle(OrderCommand::Create)
        .await
        .expect("Should be able to submit command");

    repository
        .commit_orderly(&mut root)
        .await
        .expect("Should be able to commit root");

    //
    // Check it is ready for consumption
    //

    let mut stream = subscription
        .resume()
        .await
        .expect("should resume subscription");

    let message = stream.next().await.unwrap().unwrap();

    assert_eq!(message.aggregate_id(), aggregate_id);
    assert_eq!(message.into_data(), OrderEvent::Created);
}

#[tokio::test]
async fn subscription_should_fetch_event_after_notification() {
    let TestData {
        aggregate_id,
        mut repository,
        mut root,
        subscription,
        ..
    } = bootstrap_test(false).await;

    // start stream before messages are published
    let mut stream = subscription
        .resume()
        .await
        .expect("should resume subscription");

    //
    // Append event
    //
    root.handle(OrderCommand::Create)
        .await
        .expect("Should be able to submit command");

    repository
        .commit_orderly(&mut root)
        .await
        .expect("Should be able to commit root");

    //
    // Check message is ready for consumption
    //
    let message = stream.next().await.unwrap().unwrap();

    assert_eq!(message.aggregate_id(), aggregate_id);
    assert_eq!(message.into_data(), OrderEvent::Created);
}

#[tokio::test]
async fn it_should_recover_if_postgres_notifications_buffer_gets_full() {
    let TestData {
        aggregate_type_name,
        pool,
        subscriber_name,
        ..
    } = bootstrap_test(false).await;
    let mut store = EventStoreBuilder::new(pool.clone())
        .build::<OrderEvent>(aggregate_type_name.clone().into())
        .await
        .expect("store to be created");

    let events1 = sample_events_uncommitted();
    let events2 = sample_events_uncommitted();
    let events3 = sample_events_uncommitted();

    //
    // Setup subscription with very limited capacity for buffers
    //
    let subscription = SubscriptionBuilder::default()
        .with_notification_buffer_capacity(1)
        .with_event_buffer_capacity(events1.len() - 1)
        .aggregate_build::<OrderEvent>(subscriber_name.into(), &aggregate_type_name, pool.clone())
        .await
        .unwrap();

    // start stream before messages are published
    let stream = subscription
        .resume()
        .await
        .expect("should resume subscription");

    //
    // Append events and fill buffers
    //
    let aggregate_id = Uuid::new_v4();

    // this notification should be consumed as the first one ok
    store
        .commit(aggregate_id, CommitOrder::None, &events1)
        .await
        .expect("Failed appending events");

    // this notification should make notification buffer congested
    store
        .commit(aggregate_id, CommitOrder::None, &events2)
        .await
        .expect("Failed appending events");

    // this notification should await until buffers' congestion eases
    store
        .commit(aggregate_id, CommitOrder::None, &events3)
        .await
        .expect("Failed appending events");

    // buffer backpressure should be applied
    yield_now().await;

    let received = stream.take(9).map(|x| x.unwrap()).collect::<Vec<_>>().await;

    assert_eq!(received.len(), 9, "It should receive all events");
}

#[tokio::test]
async fn handles_checkpoints_ok() {
    let TestData {
        mut repository,
        mut root,
        subscription,
        subscriber_name,
        pool,
        ..
    } = bootstrap_test(false).await;

    let mut stream = subscription
        .resume()
        .await
        .expect("should resume subscription");

    //
    // Append event
    //
    root.handle(OrderCommand::Create)
        .await
        .expect("Should be able to submit command");

    repository
        .commit_orderly(&mut root)
        .await
        .expect("Should be able to commit root");

    let message = stream.next().await.unwrap().unwrap();

    subscription.ack(message.sequence()).await.unwrap();

    let saved_offset =
        sqlx::query_file_scalar!("tests/queries/subscription_offset.sql", &subscriber_name)
            .fetch_one(&pool)
            .await
            .unwrap();

    assert_eq!(message.sequence(), saved_offset.unwrap());
}

#[tokio::test]
async fn it_should_save_last_checkpoint() {
    let TestData {
        mut repository,
        mut root,
        subscription,
        subscriber_name,
        pool,
        ..
    } = bootstrap_test(false).await;

    let stream = subscription.resume();

    //
    // Append event
    //
    root.handle(OrderCommand::Create)
        .await
        .expect("Should be able to submit command");
    root.handle(OrderCommand::Cancel)
        .await
        .expect("Should be able to submit command");

    repository
        .commit_orderly(&mut root)
        .await
        .expect("Should be able to commit root");

    let messages = stream
        .await
        .expect("should resume subscription")
        .take(2)
        .collect::<Vec<_>>()
        .await;
    let last_message = messages.last().unwrap().as_ref().unwrap();

    subscription.ack(last_message.sequence()).await.unwrap();

    let saved_offset =
        sqlx::query_file_scalar!("tests/queries/subscription_offset.sql", &subscriber_name)
            .fetch_one(&pool)
            .await
            .unwrap();

    assert_eq!(last_message.sequence(), saved_offset.unwrap());
}

#[tokio::test]
async fn it_should_handle_first_try_ack() {
    let TestData {
        mut repository,
        mut root,
        subscription,
        subscriber_name,
        pool,
        ..
    } = bootstrap_test(false).await;

    let mut stream = subscription
        .resume()
        .await
        .expect("should resume subscription");

    //
    // Append event
    //
    root.handle(OrderCommand::Create)
        .await
        .expect("Should be able to submit command");

    repository
        .commit_orderly(&mut root)
        .await
        .expect("Should be able to commit root");

    let message = stream.next().await.unwrap().unwrap();

    subscription.try_ack(message.sequence()).await.unwrap();

    let saved_offset =
        sqlx::query_file_scalar!("tests/queries/subscription_offset.sql", &subscriber_name)
            .fetch_one(&pool)
            .await
            .unwrap();

    assert_eq!(message.sequence(), saved_offset.unwrap());
}

#[tokio::test]
#[should_panic(expected = "checkpoint out of order")]
async fn it_should_return_err_for_out_of_order_try_ack() {
    let TestData {
        mut repository,
        mut root,
        subscription,
        ..
    } = bootstrap_test(false).await;

    let stream = subscription.resume();

    //
    // Append event
    //
    root.handle(OrderCommand::Create)
        .await
        .expect("Should be able to submit command");
    root.handle(OrderCommand::Cancel)
        .await
        .expect("Should be able to submit command");

    repository
        .commit_orderly(&mut root)
        .await
        .expect("Should be able to commit root");

    let messages = stream
        .await
        .expect("should resume subscription")
        .take(2)
        .collect::<Vec<_>>()
        .await;
    let last_message = messages.last().unwrap().as_ref().unwrap();

    subscription.try_ack(last_message.sequence()).await.unwrap();
}

#[tokio::test]
#[should_panic(expected = "checkpoint behind current offset")]
async fn it_should_return_err_for_already_submitted_checkpoint() {
    let TestData {
        mut repository,
        mut root,
        subscription,
        ..
    } = bootstrap_test(false).await;

    let mut stream = subscription
        .resume()
        .await
        .expect("should resume subscription");

    //
    // Append event
    //
    root.handle(OrderCommand::Create)
        .await
        .expect("Should be able to submit command");
    root.handle(OrderCommand::Cancel)
        .await
        .expect("Should be able to submit command");

    repository
        .commit_orderly(&mut root)
        .await
        .expect("Should be able to commit root");

    let first = stream.next().await.unwrap().unwrap();
    let last = stream.next().await.unwrap().unwrap();

    subscription.try_ack(first.sequence()).await.unwrap();

    subscription.try_ack(last.sequence()).await.unwrap();

    subscription.try_ack(first.sequence()).await.unwrap();
}

#[tokio::test]
#[should_panic(
    expected = "update or delete on table aggregate_subscription violates foreign key constraint"
)]
async fn it_should_return_err_for_non_existent_offset_when_trying_checkpoint() {
    let TestData {
        mut repository,
        mut root,
        subscription,
        ..
    } = bootstrap_test(false).await;

    let mut stream = subscription
        .resume()
        .await
        .expect("should resume subscription");

    root.handle(OrderCommand::Create)
        .await
        .expect("Should be able to submit command");

    repository
        .commit_orderly(&mut root)
        .await
        .expect("Should be able to commit root");

    let event = stream.next().await.unwrap().unwrap();

    subscription.try_ack(event.sequence() + 1).await.unwrap();
}
