use crate::error::{Error, Result};
use flume::Receiver;
use serde::Deserialize;
use sqlx::postgres::PgListener;
use sqlx::PgPool;
use std::borrow::Cow;
use std::fmt::Debug;
use uuid::Uuid;

#[derive(Debug, Deserialize)]
pub struct EventNotification {
    pub aggregate_id: Uuid,
    pub sequence_end: i64,
    pub sequence_prior: Option<i64>,
    pub sequence_start: i64,
    pub time_end: i64,
    pub time_start: i64,
    pub total: usize,
}

#[derive(Debug)]
pub struct Subscriber {
    buffer_capacity: usize,
    channel: Cow<'static, str>,
    pool: PgPool,
}

impl Subscriber {
    pub fn new(buffer_capacity: usize, channel: Cow<'static, str>, pool: PgPool) -> Self {
        Self {
            buffer_capacity,
            channel,
            pool,
        }
    }

    /// PgListener is an unbounded mpsc channel
    /// This subscriber:
    ///   - transforms payload
    ///   - notifies when connection is disrupted
    ///   - handles backpressure
    pub async fn listen(&self) -> Result<Receiver<Result<EventNotification>>> {
        let channel = self.channel.as_ref();
        let (push_tx, push_rx) = flume::bounded(self.buffer_capacity);
        let mut listener = PgListener::connect_with(&self.pool).await?;
        listener.listen(channel).await?;
        info!(channel, "Listening for notifications");
        let capacity = self.buffer_capacity;

        tokio::spawn(async move {
            loop {
                let message = match listener.try_recv().await {
                    Ok(Some(notification)) => {
                        debug!(?notification, "Received LISTEN notification");

                        serde_json::from_str::<EventNotification>(notification.payload())
                            .map_err(Error::from)
                    }
                    Ok(None) => Err(Error::ListenerDisconnected),
                    Err(error) => Err(Error::from(error)),
                };

                let buffer_len = push_tx.len();

                if push_tx.is_full() {
                    error!(
                        capacity,
                        "Postgres notification channel is full. Applying backpressure."
                    );
                } else if (buffer_len as f32) > (capacity as f32) * 0.66 {
                    warn!(
                        capacity,
                        buffer_len, "Congestion building up in postgres notification channel"
                    );
                }

                if let Err(error) = push_tx.send_async(message).await {
                    error!(?error, "All push notification receivers have been dropped");
                }
            }
        });

        Ok(push_rx)
    }
}
