use crate::error::{Error, Result};
use crate::event::EventRow;
use crate::TimescaleEventPayload;
use timesource_core::event::Persisted;
use futures::stream::BoxStream;
use futures::{StreamExt, TryStreamExt};
use serde::Deserialize;
use sqlx::PgPool;
use std::borrow::Cow;
use std::convert::TryFrom;
use std::fmt::Debug;
use std::marker::PhantomData;

use super::ReaderStore;
use super::StoreData;

#[derive(Clone, Debug)]
pub struct AggregateReaderStore<Event>
where
    Event: Clone + TimescaleEventPayload + Send + Sync + Debug,
    for<'de> Event: Deserialize<'de>,
{
    aggregate_type_id: i32,
    subscription_name: Cow<'static, str>,
    pool: PgPool,
    _event: PhantomData<Event>,
}

impl<Event> AggregateReaderStore<Event>
where
    Event: Clone + TimescaleEventPayload + Send + Sync + Debug,
    for<'de> Event: Deserialize<'de>,
{
    pub fn new(aggregate_type_id: i32, subscription_name: Cow<'static, str>, pool: PgPool) -> Self {
        Self {
            aggregate_type_id,
            subscription_name,
            pool,
            _event: PhantomData,
        }
    }
}

#[async_trait]
impl<'a, Event> ReaderStore<'a> for AggregateReaderStore<Event>
where
    Event: Clone + TimescaleEventPayload + Send + Sync + Debug,
    for<'de> Event: Deserialize<'de>,
{
    type Event = Event;

    #[tracing::instrument]
    fn events_after_offset(&self) -> BoxStream<'_, StoreData<Event>> {
        let aggregate_type_id = &self.aggregate_type_id;

        sqlx::query_file_as!(
            EventRow,
            "queries/subscription/events_after_offset.sql",
            aggregate_type_id,
        )
        .fetch(&self.pool)
        .map_err(Error::from)
        .map(move |x| x.and_then(|x| Persisted::try_from(x).map_err(Error::from)))
        .boxed()
    }

    #[tracing::instrument]
    fn events_after(&self, offset: i64, limit: usize) -> BoxStream<'_, StoreData<Event>> {
        let aggregate_type_id = &self.aggregate_type_id;

        sqlx::query_file_as!(
            EventRow,
            "queries/subscription/events_after_offset_limit.sql",
            aggregate_type_id,
            offset,
            limit as i32
        )
        .fetch(&self.pool)
        .map_err(Error::from)
        .map(move |x| x.and_then(|x| Persisted::try_from(x).map_err(Error::from)))
        .boxed()
    }

    #[tracing::instrument]
    fn events_range(
        &self,
        later_than: i64, // >
        until: i64,      // <=
    ) -> BoxStream<'_, StoreData<Event>> {
        let aggregate_type_id = &self.aggregate_type_id;

        sqlx::query_file_as!(
            EventRow,
            "queries/subscription/events_range.sql",
            aggregate_type_id,
            later_than,
            until
        )
        .fetch(&self.pool)
        .map_err(Error::from)
        .map(move |x| x.and_then(|x| Persisted::try_from(x).map_err(Error::from)))
        .boxed()
    }

    #[tracing::instrument]
    async fn save_offset(&self, offset: i64) -> Result<()> {
        sqlx::query_file!(
            "queries/subscription/checkout_unchecked.sql",
            self.subscription_name.as_ref(),
            offset
        )
        .execute(&self.pool)
        .await
        .map(|_| {})
        .map_err(Error::from)
    }

    #[tracing::instrument]
    async fn try_save_offset(&self, offset: i64) -> Result<()> {
        sqlx::query_file!(
            "queries/subscription/checkout.sql",
            self.subscription_name.as_ref(),
            offset
        )
        .execute(&self.pool)
        .await
        .map(|_| {})
        .map_err(Error::from)
    }
}
