//! Contains the [Repository pattern] implementation for [`AggregateRoot`]
//! instances.
//!
//! Check out [`Repository`] for more information.
//!
//! [Repository pattern]: https://docs.microsoft.com/en-us/dotnet/architecture/microservices/microservice-ddd-cqrs-patterns/infrastructure-persistence-layer-design#the-repository-pattern

use chrono::Utc;
use futures::stream::TryStreamExt;
use uuid::Uuid;

use crate::aggregate::{Aggregate, AggregateRoot};
use crate::error::{Error, Result};
use crate::store::{CommitOrder, EventStore};

#[derive(Clone)]
pub struct Repository<A, S>
where
    A: Aggregate + 'static,
    S: EventStore<Event = A::Event>,
{
    aggregate: A,
    store: S,
}

impl<A, S> Repository<A, S>
where
    A: Aggregate,
    S: EventStore<Event = A::Event>,
{
    pub fn new(aggregate: A, store: S) -> Self {
        Repository { aggregate, store }
    }

    pub async fn get(&self, aggregate_id: Uuid) -> Result<AggregateRoot<'_, A>> {
        let (utc, state) = self
            .store
            .aggregate_stream(aggregate_id)
            .await?
            .try_fold(
                (Utc::now(), None),
                |(_, mut fold_state), event| async move {
                    let utc = event.utc();
                    let aggregate_id = event.aggregate_id();
                    let data = &event.into_data();
                    let state = match fold_state {
                        None => match A::apply_first(aggregate_id, data, utc) {
                            Ok(state) => Some(state),
                            Err(error) => return Err(Error::Aggregate(Box::new(error))),
                        },
                        Some(ref mut state) => {
                            if let Err(error) = A::apply_next(state, data, utc) {
                                return Err(Error::Aggregate(Box::new(error)));
                            };

                            fold_state
                        }
                    };

                    Ok((utc, state))
                },
            )
            .await?;

        match state {
            Some(state) => Ok(self.aggregate.root_with_state(aggregate_id, utc, state)),
            None => Err(Error::AggregateRootNotFound),
        }
    }

    pub async fn commit_orderly(&mut self, root: &mut AggregateRoot<'_, A>) -> Result<()> {
        let events = root.uncommitted_events();

        if !events.is_empty() {
            let order = match root.last_commit_utc() {
                Some(utc) => CommitOrder::Following(utc),
                None => CommitOrder::First,
            };

            self.store.commit(root.id(), order, events).await?;
            root.commit();
        }

        Ok(())
    }

    pub async fn commit_unorderly(&mut self, root: &mut AggregateRoot<'_, A>) -> Result<()> {
        let events = root.uncommitted_events();

        if !events.is_empty() {
            self.store
                .commit(root.id(), CommitOrder::None, events)
                .await?;
        }

        Ok(())
    }

    pub async fn remove(&mut self, aggregate_id: Uuid) -> Result<()> {
        self.store.remove(aggregate_id).await
    }
}
