mod common;

use chrono::Utc;
use common::order::OrderCommand;
use timesource::aggregate::UncommittedEvent;
use timesource::store::{CommitOrder, EventStore, EventStoreBuilder};
use timesource::{Aggregate, TimescaleEvent, TimescaleEventPayload};
use futures::TryStreamExt;
use serde::{Deserialize, Serialize};
use serde_json::Value as Json;
use sqlx::FromRow;
use uuid::Uuid;

use crate::common::data::{
    bootstrap_test, sample_commands, sample_events, sample_events_uncommitted, TestData,
};
use crate::common::order::{OrderAggregate, OrderEvent};

#[macro_use]
extern crate test_case;

#[derive(Debug, FromRow)]
struct EventRow {
    aggregate_type_id: i32,
    name: String,
    payload: Option<Json>,
    time: i64,
}

#[derive(Debug, FromRow)]
struct EventRowName {
    aggregate_type_id: i32,
    id: i32,
    name: String,
}

#[test_case(true ; "ordered")]
#[test_case(false ; "unordered")]
#[tokio::test]
async fn should_normalise_and_store_events_efficiently(ordered: bool) {
    let TestData {
        aggregate_id,
        pool,
        mut repository,
        mut root,
        aggregate_type_id,
        ..
    } = bootstrap_test(false).await;

    for cmd in sample_commands() {
        root.handle(cmd)
            .await
            .expect("Should be able to submit command");
    }

    if ordered {
        repository
            .commit_orderly(&mut root)
            .await
            .expect("Should be able to commit orderly root");
    } else {
        repository
            .commit_unorderly(&mut root)
            .await
            .expect("Should be able to commit un-orderly root");
    }

    let event_rows = sqlx::query_file_as!(
        EventRow,
        "tests/queries/get_aggregate_events.sql",
        aggregate_id
    )
    .fetch_all(&pool)
    .await
    .expect("client query for events");

    if let [struct_unit_row, struct_fields_row, struct_tuple_row] = &event_rows[..] {
        //
        // Storage of JSON payloads
        //

        assert!(
            struct_unit_row.payload.is_none(),
            "Unit structs don't need JSONB stored"
        );
        assert!(
            struct_fields_row
                .payload
                .as_ref()
                .map(|x| x.is_object())
                .unwrap_or(false),
            "Events with struct with named fields should have JSONB"
        );
        assert!(
            struct_tuple_row
                .payload
                .as_ref()
                .map(|x| x.is_string())
                .unwrap_or(false),
            "Events with scalar tuple structs should have a JSONB scalar stored"
        );
    } else {
        panic!("Unexpected event_rows");
    }

    //
    // Storage of Event names
    //

    let event_name_data = sqlx::query_file_as!(
        EventRowName,
        "tests/queries/get_event_names.sql",
        event_rows[0].aggregate_type_id
    )
    .fetch_all(&pool)
    .await
    .expect("client query for event name data");

    let event_names = event_name_data
        .iter()
        .map(|x| (x.name.as_str()))
        .collect::<Vec<&str>>();

    assert_eq!(
        event_names,
        sample_events().iter().map(|x| x.name()).collect::<Vec<_>>(),
        "Data normalization for event names is not right"
    );

    for event_name in event_name_data {
        assert_eq!(
            event_name.aggregate_type_id, aggregate_type_id,
            "Invalid foreign key for aggregate root id"
        );
    }
}

#[tokio::test]
async fn different_aggregate_types_can_share_id() {
    let TestData {
        pool,
        aggregate_type_name,
        ..
    } = bootstrap_test(false).await;

    let shared_id = Uuid::new_v4();

    #[derive(Debug, Clone, Serialize, Deserialize, TimescaleEvent)]
    enum Store2Event {
        Something,
    }

    let mut store = EventStoreBuilder::new(pool.clone())
        .build::<OrderEvent>(aggregate_type_name.into())
        .await
        .expect("store to be created");

    let mut store2 = EventStoreBuilder::new(pool.clone())
        .build::<Store2Event>("orders2".into())
        .await
        .expect("store to be created");

    let events1 = sample_events_uncommitted();

    let events2 = vec![Store2Event::Something; events1.len()]
        .into_iter()
        .map(|data| UncommittedEvent {
            utc: Utc::now(),
            data,
        })
        .collect::<Vec<_>>();

    store
        .commit(shared_id, CommitOrder::None, &events1)
        .await
        .expect("Failed appending events");

    store2
        .commit(shared_id, CommitOrder::None, &events2)
        .await
        .expect("Failed appending events");

    let read_events = store
        .aggregate_stream(shared_id)
        .await
        .expect("failed to create first stream")
        .try_collect::<Vec<_>>()
        .await
        .expect("failed to collect first stream of events from subscription");

    let read_events2 = store2
        .aggregate_stream(shared_id)
        .await
        .expect("failed to create second stream")
        .try_collect::<Vec<_>>()
        .await
        .expect("failed to collect second stream of events from subscription");

    for (i, event) in read_events.iter().enumerate() {
        assert_eq!(
            event.aggregate_id(),
            read_events2[i].aggregate_id(),
            "Expected event source ids to be the same between stores"
        );
    }
}

#[tokio::test]
#[should_panic(expected = "list of events can't be empty")]
async fn return_err_given_no_events() {
    let TestData {
        pool,
        aggregate_type_name,
        ..
    } = bootstrap_test(false).await;

    let mut store = EventStoreBuilder::new(pool.clone())
        .build::<OrderEvent>(aggregate_type_name.into())
        .await
        .expect("store to be created");

    store
        .commit(Uuid::new_v4(), CommitOrder::None, &[])
        .await
        .unwrap();
}

#[tokio::test]
#[should_panic(expected = "AlreadyCreated")]
async fn should_propagate_aggregate_err_when_handling_command() {
    let TestData {
        mut repository,
        mut root,
        ..
    } = bootstrap_test(false).await;

    // these are invalid events
    root.handle(OrderCommand::Create).await.unwrap();
    root.handle(OrderCommand::Create).await.unwrap();

    repository.commit_unorderly(&mut root).await.unwrap();
}

// Unordered events can't happen when using one AggregateRoot
// However, it can happen on distributed systems with a race condition
#[tokio::test]
async fn should_reorder_events_when_committed_unorderly() {
    let TestData {
        aggregate_id,
        pool,
        mut repository,
        root,
        ..
    } = bootstrap_test(false).await;

    let mut system_a_root = root;
    let mut system_b_root = OrderAggregate.root(aggregate_id);

    // given system A handles commands first
    for command in sample_commands() {
        system_a_root
            .handle(command)
            .await
            .expect("Failed handling event for systemA");
    }

    // given system B handles commands later on
    for command in sample_commands() {
        system_b_root
            .handle(command)
            .await
            .expect("Failed handling event for systemA");
    }

    let total_events = {
        let system_a_events = system_a_root.uncommitted_events();
        let system_b_events = system_b_root.uncommitted_events();
        for (index, event) in system_a_events.iter().enumerate() {
            assert!(
                event.utc < system_b_events[index].utc,
                "Something is not right with this test. Events in System A should be created before B"
            );
        }
        system_a_events.len() + system_b_events.len()
    };

    // but then system B commit first
    repository
        .commit_unorderly(&mut system_b_root)
        .await
        .unwrap();

    // and system A events arrive later even though they were handled first
    repository
        .commit_unorderly(&mut system_a_root)
        .await
        .unwrap();

    let event_rows = sqlx::query_file_as!(
        EventRow,
        "tests/queries/get_aggregate_events.sql",
        aggregate_id
    )
    .fetch_all(&pool)
    .await
    .expect("client query for events");

    assert_eq!(
        event_rows.len(),
        total_events,
        "should have saved all events in db"
    );

    for (i, row) in event_rows.iter().enumerate() {
        if i > 0 {
            assert!(
                row.time > event_rows[i - 1].time,
                "Unordered events can't be inserted with a past date"
            )
        }
    }

    for (i, row) in event_rows[3..].iter().enumerate() {
        assert_eq!(
            row.time,
            event_rows[2].time + i as i64 + 1,
            "Events with a past date should be reordered automatically"
        )
    }
}

// same as before, but when committed orderly, it shouldn't allow it
#[tokio::test]
#[should_panic(expected = "invalid aggregate time provided")]
async fn return_err_for_ordered_events_which_are_not_first() {
    let TestData {
        aggregate_id,
        mut repository,
        root,
        ..
    } = bootstrap_test(false).await;

    let mut system_a_root = root;
    let mut system_b_root = OrderAggregate.root(aggregate_id);

    // given system A handles first commands first
    for command in sample_commands() {
        system_a_root
            .handle(command)
            .await
            .expect("Failed handling event for systemA");
    }

    // given system B handles first commands later on
    for command in sample_commands() {
        system_b_root
            .handle(command)
            .await
            .expect("Failed handling event for systemA");
    }

    let system_a_events = system_a_root.uncommitted_events();
    let system_b_events = system_b_root.uncommitted_events();
    for (index, event) in system_a_events.iter().enumerate() {
        assert!(
            event.utc < system_b_events[index].utc,
            "Something is not right with this test. Events in System A should be created before B"
        );
    }

    // but then system B commit first
    repository.commit_orderly(&mut system_b_root).await.unwrap();

    // and system A events arrive later so they should not be saved
    repository.commit_orderly(&mut system_a_root).await.unwrap();
}

#[tokio::test]
#[should_panic(expected = "invalid aggregate time provided")]
async fn return_err_given_unordered_events_and_existing_offset() {
    let TestData {
        aggregate_id,
        mut repository,
        root,
        ..
    } = bootstrap_test(false).await;

    let mut system_a_root = root;
    let mut system_b_root = OrderAggregate.root(aggregate_id);

    // given system A handles commands first
    for command in sample_commands() {
        system_a_root
            .handle(command)
            .await
            .expect("Failed handling event for systemA");
    }

    // given system B handles commands later on
    for command in sample_commands() {
        system_b_root
            .handle(command)
            .await
            .expect("Failed handling event for systemA");
    }

    let system_a_events = system_a_root.uncommitted_events();
    let system_b_events = system_b_root.uncommitted_events();
    for (index, event) in system_a_events.iter().enumerate() {
        assert!(
            event.utc < system_b_events[index].utc,
            "Something is not right with this test. Events in System A should be created before B"
        );
    }

    // but then system B commit first
    repository.commit_orderly(&mut system_b_root).await.unwrap();

    // and system A events arrive later so they should not be saved
    repository.commit_orderly(&mut system_a_root).await.unwrap();
}
