use chrono::Utc;
use timesource::aggregate::UncommittedEvent;
use timesource::repository::Repository;
use timesource::store::{EventStoreBuilder, TimescaleStore};
use timesource::subscription::{
    AggregateReaderStore, Subscription, SubscriptionBuilder,
};
use timesource::{Aggregate, AggregateRoot};
use sqlx::postgres::PgPoolOptions;
use sqlx::{Pool, Postgres};
use tracing_subscriber::prelude::*;
use tracing_subscriber::EnvFilter;
use uuid::Uuid;

use crate::common::order::OrderItem;

use super::order::{OrderAggregate, OrderCommand, OrderEvent};

pub const DSN: &str = "postgres://postgres@localhost/test";

use gabble::Gab;
use rand::{thread_rng, Rng};

pub struct TestData<'a> {
    pub aggregate_id: Uuid,
    pub aggregate_type_id: i32,
    pub aggregate_type_name: String,
    pub pool: Pool<Postgres>,
    pub repository: Repository<OrderAggregate, TimescaleStore<OrderEvent>>,
    pub root: AggregateRoot<'a, OrderAggregate>,
    pub subscriber_name: String,
    pub subscription: Subscription<AggregateReaderStore<OrderEvent>>,
}

pub async fn bootstrap_test<'a>(with_tracing: bool) -> TestData<'a> {
    if with_tracing {
        let filter_layer = EnvFilter::try_from_default_env()
            .or_else(|_| EnvFilter::try_new("timesource=debug"))
            .unwrap();

        tracing_subscriber::registry()
            .with(filter_layer)
            .with(tracing_subscriber::fmt::layer())
            .try_init()
            .unwrap_or_else(|_| {});
    }

    let mut rng = thread_rng();
    let aggregate_type_name: Gab = rng.gen();
    let aggregate_type_name = aggregate_type_name.to_string();
    let subscriber_name: Gab = rng.gen();
    let subscriber_name = subscriber_name.to_string();
    let aggregate_id = Uuid::new_v4();

    let pool = PgPoolOptions::new()
        .connect(DSN)
        .await
        .expect("being able to create Pg pool");

    let store = EventStoreBuilder::new(pool.clone())
        .build::<OrderEvent>(aggregate_type_name.clone().into())
        .await
        .expect("store to be created");

    let subscription = SubscriptionBuilder::default()
        .aggregate_build(
            subscriber_name.clone().into(),
            &aggregate_type_name,
            pool.clone(),
        )
        .await
        .unwrap();

    let aggregate_type_id = store.aggregate_type_id;
    let repository = Repository::new(OrderAggregate, store);
    let root = OrderAggregate.root(aggregate_id);

    TestData {
        aggregate_id,
        aggregate_type_id,
        aggregate_type_name,
        repository,
        root,
        pool,
        subscriber_name,
        subscription,
    }
}

#[allow(dead_code)]
pub fn sample_commands() -> Vec<OrderCommand> {
    vec![
        OrderCommand::Create,
        OrderCommand::AddItem {
            item: OrderItem {
                item_sku: "sku-123".into(),
                quantity: 1,
                price: 12,
            },
        },
        OrderCommand::Empty("for no reason".to_string()),
    ]
}

pub fn sample_events() -> Vec<OrderEvent> {
    vec![
        OrderEvent::Created,
        OrderEvent::ItemAdded {
            item: OrderItem {
                item_sku: "sku-123".into(),
                quantity: 1,
                price: 12,
            },
        },
        OrderEvent::Emptied("for no reason".into()),
    ]
}

#[allow(dead_code)]
pub fn sample_events_uncommitted() -> Vec<UncommittedEvent<OrderEvent>> {
    sample_events()
        .into_iter()
        .map(|data| UncommittedEvent {
            utc: Utc::now(),
            data,
        })
        .collect::<Vec<_>>()
}
