//! Contains a persisted implementation of the [`Subscription`] trait
//! using Postgres as the backend data source for its state.
//!
//! [`Subscription`]: ../../eventually-core/subscription/trait.Subscription.html

mod coordinator;
mod store;
mod subscriber;

use crate::error::{Error, Result};
use crate::TimescaleEventPayload;
use coordinator::coordinate_event_channel;
use futures::stream::{BoxStream, StreamExt};
use serde::Deserialize;
use sqlx::PgPool;
use std::borrow::Cow;
use std::fmt::Debug;
use std::time::Duration;
use timesource_core::event::Persisted;

pub use self::store::aggregate::*;

use self::store::ReaderStore;
use self::subscriber::Subscriber;

pub struct SubscriptionBuilder {
    event_buffer_capacity: usize,
    notification_buffer_capacity: usize,
    polling_freq: Duration,
}

impl Default for SubscriptionBuilder {
    fn default() -> Self {
        Self {
            event_buffer_capacity: 100,
            notification_buffer_capacity: 100,
            polling_freq: Duration::from_secs(60),
        }
    }
}

impl SubscriptionBuilder {
    /// Sets the frequency of long polling for new events.
    ///
    /// When new events are being inserted in the store at a rate lower than this value,
    /// polling will never occur
    ///
    /// Long polling is only a backup mechanism when postgres LISTEN notifications are lost
    /// (e.g. temporary network failure). Therefore, it shouldn't be set too low, or else
    /// polling will consume resources from the database server unnecessarily
    ///
    /// Default value is 1 minute
    ///
    /// [`SubscriptionBuilder`]: struct.SubscriptionBuilder.html
    pub fn with_backup_polling_frequency(mut self, duration: Duration) -> Self {
        self.polling_freq = duration;
        self
    }

    /// Sets the capacity of the event buffer.
    ///
    /// Subscription will store a buffer of events for better performance.
    /// After bootstrap and when the buffer reaches its capacity, the Subscription will pause
    /// until consumer has caught up.
    ///
    /// Defaults to 100.
    ///
    /// [`SubscriptionBuilder`]: struct.SubscriptionBuilder.html
    pub fn with_event_buffer_capacity(mut self, capacity: usize) -> Self {
        self.event_buffer_capacity = capacity;
        self
    }

    pub fn with_notification_buffer_capacity(mut self, capacity: usize) -> Self {
        self.notification_buffer_capacity = capacity;
        self
    }

    /// Creates a new [`Subscription`] with the specified name for the `aggregate_type_name`
    /// if it doesn't exists already.
    ///
    pub async fn aggregate_build<Event>(
        self,
        subscription_name: Cow<'static, str>,
        aggregate_type_name: &str,
        pool: PgPool,
    ) -> Result<Subscription<AggregateReaderStore<Event>>>
    where
        Event: Clone + TimescaleEventPayload + 'static + Send + Sync + Debug,
        for<'de> Event: Deserialize<'de>,
    {
        let aggregate_type_id =
            sqlx::query_file_scalar!("queries/aggregate_type/id.sql", aggregate_type_name)
                .fetch_one(&pool)
                .await?
                .ok_or_else(|| Error::InvalidData("Unable to get aggregate type id".into()))?;

        //create subscription if it doesn't exist
        let offset = sqlx::query_file_scalar!(
            "queries/subscription/offset.sql",
            subscription_name.as_ref(),
            aggregate_type_id
        )
        .fetch_one(&pool)
        .await?;

        debug!(aggregate_type_id, ?offset, "Subscription state");

        let store = AggregateReaderStore::new(aggregate_type_id, subscription_name, pool.clone());

        let subscriber = Subscriber::new(
            self.notification_buffer_capacity,
            aggregate_type_id.to_string().into(),
            pool,
        );

        Ok(Subscription {
            event_buffer_capacity: self.event_buffer_capacity,
            polling_freq: self.polling_freq,
            store,
            subscriber,
        })
    }

    // TODO: same idea but for aggregate root only
    // pub async fn aggregate_root_build<Event>(
    //     self,
    //     subscription_name: Cow<'static, str>,
    //     aggregate_root_id: Uuid,
    //     pool: PgPool,
    // ) -> Result<Subscription<Event, AggregateReaderStore<Event>>>
    // }
}

pub struct Subscription<Store>
where
    for<'de> Store: ReaderStore<'de> + Clone + Send + Sync + 'static + Debug,
{
    event_buffer_capacity: usize,
    polling_freq: Duration,
    subscriber: Subscriber,
    store: Store,
}

impl<Store> Subscription<Store>
where
    for<'de> Store: ReaderStore<'de> + Clone + Send + Sync + 'static + Debug,
{
    pub async fn resume(
        &self,
    ) -> Result<BoxStream<'_, Result<Persisted<<Store as ReaderStore<'static>>::Event>>>> {
        let (buffer_tx, buffer_rx) = flume::bounded(self.event_buffer_capacity);

        // start listening for new messages before building state from last offset
        let pg_notification_rx = self.subscriber.listen().await?;

        let output = buffer_rx.clone();
        let polling_freq = self.polling_freq;
        let store = self.store.clone();
        tokio::spawn(async move {
            loop {
                let handle = tokio::spawn(coordinate_event_channel(
                    polling_freq,
                    pg_notification_rx.clone(),
                    store.clone(),
                    buffer_tx.clone(),
                    buffer_rx.clone(),
                ))
                .await;

                if let Err(error) = handle {
                    error!(?error, "events_fetch_task panicked. Re-spawning.");
                }
            }
        });

        Ok(output.into_stream().boxed())
    }

    pub async fn ack(&self, offset: i64) -> Result<()> {
        self.store.save_offset(offset).await.map_err(Error::from)
    }

    pub async fn try_ack(&self, offset: i64) -> Result<()> {
        self.store
            .try_save_offset(offset)
            .await
            .map_err(Error::from)
    }
}
