use super::store::{ReaderStore, StoreData};
use super::subscriber::EventNotification;
use crate::error::{Error, Result};
use backoff::ExponentialBackoff;
use flume::{Receiver, Sender};
use futures::stream::{BoxStream, StreamExt};
use std::fmt::Debug;
use std::time::Duration;
use timesource_core::Persisted;
use tokio::select;
use tokio::time::sleep;
use tracing::Instrument;

fn is_error_recoverable(error: &Error) -> bool {
    if let Error::Sqlx(error) = error {
        matches!(
            error,
            sqlx::Error::Database(_)
                | sqlx::Error::Io(_)
                | sqlx::Error::Tls(_)
                | sqlx::Error::PoolTimedOut
                | sqlx::Error::PoolClosed
                | sqlx::Error::WorkerCrashed
        )
    } else {
        false
    }
}

#[derive(Debug)]
enum FetchStreamOp {
    None,
    All,
    From(i64),
    Range(i64, i64),
}

impl FetchStreamOp {
    async fn process<Event>(
        &self,
        stream: Option<BoxStream<'_, StoreData<Event>>>,
        sink: Sender<Result<Persisted<Event>>>,
    ) -> std::result::Result<Option<i64>, backoff::Error<Option<i64>>> {
        let mut next_offset = None;

        if let Some(mut stream) = stream {
            loop {
                match stream.next().await {
                    None => {
                        debug!("No more events to process for now");
                        break;
                    }
                    Some(Ok(event)) => {
                        next_offset = Some(event.sequence());
                        sink.send_async(Ok(event)).await.unwrap();
                    }
                    Some(Err(error)) => {
                        // is next_offset is some, don't repeat the operation, or else events may get delivered more than once
                        let output = if !is_error_recoverable(&error) || next_offset.is_some() {
                            debug!(?error, "Unable to get data from store for subscription");
                            sink.send_async(Err(error)).await.unwrap();
                            backoff::Error::Permanent(next_offset)
                        } else {
                            debug!(
                                ?error,
                                "Recoverable error detected when consuming stream. Retrying"
                            );
                            backoff::Error::Transient(next_offset)
                        };
                        return Err(output);
                    }
                }
            }
        }

        Ok(next_offset)
    }
}

///
/// This process does 3 things:
/// 1. Catch up with events since last offset on bootstrap (if any)
/// 2. Listen for notifications of new events
/// 3. Poll for events when:
///     - an anomaly is detected and self-healing is required
///     - the notification channel has been disconnected
///     - a heartbeat may be a good idea after no events have been received for `default_poller_frequency`
pub(super) async fn coordinate_event_channel<Store>(
    default_poller_frequency: Duration,
    pg_notification_rx: Receiver<Result<EventNotification>>,
    store: Store,
    buffer_tx: Sender<Result<Persisted<Store::Event>>>,
    buffer_rx: Receiver<Result<Persisted<Store::Event>>>,
) where
    Store: ReaderStore<'static>,
{
    info!("Catching up with events published since last committed offset");

    let buffer_capacity = buffer_rx.capacity().unwrap_or_default();
    let mut resume_stream = store.events_after_offset();
    let mut last_known_offset = None;
    let mut prior_events_total = 0;

    while let Some(r) = resume_stream.next().await {
        let event_sequence = r.as_ref().map(|x| x.sequence()).ok();

        buffer_tx.send_async(r).await.unwrap();

        prior_events_total += 1;
        if let Some(offset) = event_sequence {
            last_known_offset = Some(offset);
        }
    }

    info!(
        last_known_offset,
        total_processed = prior_events_total,
        "Caught up with events published before bootstrap. Waiting for next event."
    );

    let mut poller_frequency = default_poller_frequency;

    loop {
        let stream_op_kind = async {
            let poller = sleep(poller_frequency);
            let pg_notification = pg_notification_rx.recv_async();

            select! {
                _ = poller => {
                    debug!(?poller_frequency, "Polling for events in case there are network issues");
                    match last_known_offset {
                        Some(offset) => FetchStreamOp::From(offset),
                        None => FetchStreamOp::All
                    }
                }
                payload = pg_notification => match payload {
                    Ok(Ok(notification)) => {
                        if poller_frequency != default_poller_frequency {
                            info!("Subscription connection re-established. Disabling polling as first option");
                            poller_frequency = default_poller_frequency
                        }

                        let start = notification.sequence_start;
                        let end = notification.sequence_end;
                        let sequence_prior = notification.sequence_prior;

                        let is_invalid = end < start || sequence_prior.is_none() && last_known_offset.is_some();
                        let already_processed = last_known_offset.map(|x| start < x && end <= x).unwrap_or_default();
                        let is_partially_late = last_known_offset.map(|x| start <= x).unwrap_or_default();
                        let missed_beginning_of_seq = last_known_offset.is_none() && sequence_prior.is_some();
                        let follows_seq = sequence_prior.and_then(|x| last_known_offset.map(|y| x == y)).unwrap_or(true);

                        if is_invalid {
                            error!(?notification, "Invalid notification");
                            FetchStreamOp::None
                        } else if already_processed {
                            warn!(?notification, ?last_known_offset, "Received postgres notification for events which have already been processed");
                            FetchStreamOp::None
                        } else if is_partially_late {
                            // This shouldn't happen if events are committed as part of a transaction
                            error!(?notification, ?last_known_offset, "Invalid notification. Overlapping postgres notification for events which are partially processed");
                            FetchStreamOp::None
                        } else if missed_beginning_of_seq {
                            error!(?notification, ?last_known_offset, "Unexpected postgres notification. Missed beginning of sequence.");
                            FetchStreamOp::Range(std::i64::MIN, end)
                        } else if !follows_seq {
                            error!(?notification, ?last_known_offset, "It seems some postgres notifications went missing. Rewinding and catching up.");
                            FetchStreamOp::Range(last_known_offset.unwrap(), end)
                        } else {
                            debug!(?notification, "Postgres LISTEN notification received");
                            // last_known_offset and sequence_prior are the same
                            // but both may be None if it's a notification for the first events of the aggregate root
                            let offset = sequence_prior.unwrap_or(std::i64::MIN);
                            FetchStreamOp::Range(offset, end)
                        }
                    },
                    Ok(Err(error)) => {
                        match error {
                            Error::ListenerDisconnected => {
                                error!(?error, "Subscription connection interrupted. Switching to polling");
                                poller_frequency = Duration::from_secs(3);
                                FetchStreamOp::None
                            },
                            _ => {
                                error!(?error, "Unexpected subscription internal error");
                                FetchStreamOp::None
                            }
                        }
                    }
                    Err(error) => {
                        error!(?error, "pg_notification channel dropped");
                        FetchStreamOp::None
                    },
                }
            }
        }
        .instrument(info_span!("trigger_await", last_known_offset))
        .await;

        debug!(operation = ?stream_op_kind, "New operation");

        async {
            let make_stream = || match stream_op_kind {
                FetchStreamOp::None => None,
                FetchStreamOp::All => Some(store.events_after_offset()),
                FetchStreamOp::From(offset) => {
                    let available_capacity = buffer_capacity - buffer_rx.len();
                    Some(store.events_after(offset, available_capacity))
                }
                FetchStreamOp::Range(start, end) => Some(store.events_range(start, end)),
            };

            let result = backoff::future::retry(ExponentialBackoff::default(), || {
                stream_op_kind.process(make_stream(), buffer_tx.clone())
            })
            .await;

            match result {
                Ok(None) => {}
                Ok(Some(next_offset)) => {
                    last_known_offset = Some(next_offset);
                }
                Err(next_offset) => {
                    last_known_offset = next_offset;
                }
            }
        }
        .instrument(info_span!("subscription_event_fetch"))
        .await;
    }
}

#[cfg(test)]
mod tests {
    use std::convert::TryFrom;

    use super::super::store::*;
    use super::*;
    use crate::{self as timesource};
    use crate::{TimescaleEvent, TimescaleEventPayload};
    use chrono::Utc;
    use futures::stream::{self, BoxStream};
    use mockall::mock;
    use mockall::predicate::{self, *};
    use serde::{Deserialize, Serialize};
    use timesource_core::event::EventRow;
    use timesource_core::event::Persisted;
    use tokio::task::yield_now;
    use uuid::Uuid;

    #[derive(Debug, Clone, Serialize, Deserialize, PartialEq, Eq, TimescaleEvent)]
    enum StoreEvents {
        SomeEvent1,
        SomeEvent2,
    }

    fn set_panic_hook() {
        let default_panic = std::panic::take_hook();
        std::panic::set_hook(Box::new(move |info| {
            default_panic(info);
            std::process::exit(1);
        }));
    }

    mock! {
        Store {}

        #[async_trait]
        impl<'a> ReaderStore<'a> for Store
            {
                type Event = StoreEvents;

                fn events_after_offset(&self) -> BoxStream<'_, StoreData<StoreEvents>>;

                fn events_after(&self, offset: i64, limit: usize) -> BoxStream<'_, StoreData<StoreEvents>>;

                fn events_range(&self, later_than: i64, until: i64) -> BoxStream<'_, StoreData<StoreEvents>>;

                async fn save_offset(&self, offset: i64) -> Result<()>;

                async fn try_save_offset(&self, offset: i64) -> Result<()>;
            }
    }

    struct TestRunner {
        aggregate_id: Uuid,
        sequence: i64,
        event_buffer_rx: Receiver<Result<Persisted<StoreEvents>>>,
        event_buffer_tx: Sender<Result<Persisted<StoreEvents>>>,
        pg_rx: Receiver<Result<EventNotification>>,
        pg_tx: Sender<Result<EventNotification>>,
        last_offset: Option<i64>,
    }

    impl TestRunner {
        fn new() -> Self {
            set_panic_hook();
            let aggregate_id = Uuid::new_v4();
            let (event_buffer_tx, event_buffer_rx) = flume::bounded(100);
            let (pg_tx, pg_rx) = flume::bounded(100);

            Self {
                aggregate_id,
                sequence: 0,
                event_buffer_rx,
                event_buffer_tx,
                last_offset: None,
                pg_rx,
                pg_tx,
            }
        }

        fn make_events(&mut self, inputs: Vec<StoreEvents>) -> Vec<Persisted<StoreEvents>> {
            inputs
                .into_iter()
                .map(|x| {
                    let sequence = self.sequence;
                    self.sequence += 1;
                    let name = x.name().to_string();
                    let mut value = serde_json::to_value(&x).unwrap();

                    let data = if let serde_json::Value::Object(value) = &mut value {
                        value.remove(&name)
                    } else {
                        None
                    };
                    let row = EventRow {
                        aggregate_id: self.aggregate_id,
                        sequence,
                        name,
                        payload: data,
                        time: Utc::now().timestamp_nanos(),
                    };

                    Persisted::try_from(row).unwrap()
                })
                .collect::<Vec<_>>()
        }

        fn append_events(
            &mut self,
            inputs: Vec<StoreEvents>,
            notify: bool,
        ) -> Vec<Persisted<StoreEvents>> {
            let events = self.make_events(inputs);

            for event in &events {
                let time = event.utc().timestamp_nanos();
                self.event_buffer_tx.send(Ok(event.clone())).unwrap();
                self.last_offset = Some(time);
            }

            if notify {
                let notification = EventNotification {
                    aggregate_id: self.aggregate_id,
                    sequence_start: events.first().unwrap().sequence(),
                    sequence_end: events.last().unwrap().sequence(),
                    sequence_prior: self.last_offset,
                    time_start: events[0].utc().timestamp_nanos(),
                    time_end: events.last().unwrap().utc().timestamp_nanos(),
                    total: events.len(),
                };

                self.pg_tx.send(Ok(notification)).unwrap();
            }

            events
        }

        async fn append_events_and_notify(
            &mut self,
            inputs: Vec<StoreEvents>,
        ) -> Vec<Persisted<StoreEvents>> {
            let events = self.append_events(inputs, true);
            sleep(Duration::from_millis(100)).await;
            events
        }

        fn append_events_only(&mut self, inputs: Vec<StoreEvents>) -> Vec<Persisted<StoreEvents>> {
            self.append_events(inputs, false)
        }

        fn stream_all(&self) -> BoxStream<'static, Result<Persisted<StoreEvents>>> {
            let buffer = self.event_buffer_rx.clone().drain().collect::<Vec<_>>();
            stream::iter(buffer).boxed()
        }
    }

    //
    // BOOTSTRAP
    //

    #[tokio::test]
    async fn on_spawn_streams_since_offset() {
        set_panic_hook();

        let poller_frequency = Duration::from_secs(60);
        let (_pg_tx, pg_rx) = flume::bounded(100);
        let (buffer_tx, buffer_rx) = flume::bounded::<Result<Persisted<StoreEvents>>>(100);

        let mut store = MockStore::new();
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| stream::iter(vec![]).boxed());
        store.expect_events_after().times(0);
        store.expect_events_range().times(0);

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            pg_rx,
            store,
            buffer_tx,
            buffer_rx,
        ));
        sleep(Duration::from_millis(100)).await;
        task.abort();
    }

    #[tokio::test]
    async fn on_spawn_sends_events_since_offset() {
        let poller_frequency = Duration::from_secs(60);
        let mut runner = TestRunner::new();

        let appended =
            runner.append_events_only(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);

        let mut store = MockStore::new();
        let stream_all = runner.stream_all();
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| stream_all);
        store.expect_events_after().times(0);
        store.expect_events_range().times(0);

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            runner.pg_rx.clone(),
            store,
            runner.event_buffer_tx.clone(),
            runner.event_buffer_rx.clone(),
        ));

        yield_now().await;

        let received = runner
            .event_buffer_rx
            .drain()
            .map(|x| x.unwrap())
            .collect::<Vec<_>>();

        assert_eq!(appended, received);

        task.abort();
    }

    #[tokio::test]
    async fn on_spawn_yields_when_buffer_is_full() {
        let poller_frequency = Duration::from_secs(60);
        let mut runner = TestRunner::new();
        let (buffer_tx, buffer_rx) = flume::bounded::<Result<Persisted<StoreEvents>>>(1);

        let events = runner
            .append_events_and_notify(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2])
            .await;

        let mut store = MockStore::new();
        let stream_all = runner.stream_all();
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| stream_all);
        store.expect_events_after().times(0);
        store.expect_events_range().times(0);

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            runner.pg_rx.clone(),
            store,
            buffer_tx,
            buffer_rx.clone(),
        ));

        yield_now().await;

        let time = Utc::now().timestamp_nanos();
        let notification = EventNotification {
            aggregate_id: runner.aggregate_id,
            sequence_start: events[0].sequence(),
            sequence_end: events.len() as i64,
            sequence_prior: Some(events[1].utc().timestamp_nanos()),
            time_start: time,
            time_end: time,
            total: events.len(),
        };

        // even after sending another notification it shouldn't advance
        runner.pg_tx.send(Ok(notification)).unwrap();
        assert!(buffer_rx.is_full());

        task.abort();
    }

    //
    // POLLING
    //

    #[tokio::test]
    async fn on_polling_streams_from_last_known_offset() {
        let poller_frequency = Duration::from_millis(50);
        let mut runner = TestRunner::new();

        let events =
            runner.append_events_only(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);

        let mut store = MockStore::new();
        let stream_all = runner.stream_all();
        let events_after = runner.stream_all();
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| stream_all);
        store
            .expect_events_after()
            .with(
                predicate::eq(events[1].sequence()),
                predicate::eq(runner.event_buffer_tx.capacity().unwrap() - 2),
            )
            .times(1)
            .return_once(|_, _| events_after);
        store.expect_events_range().times(0);

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            runner.pg_rx.clone(),
            store,
            runner.event_buffer_tx.clone(),
            runner.event_buffer_rx.clone(),
        ));

        sleep(Duration::from_millis(51)).await;

        task.abort();
    }

    #[tokio::test]
    async fn on_polling_streams_all_if_offset_not_known() {
        let poller_frequency = Duration::from_millis(50);
        let runner = TestRunner::new();

        let mut store = MockStore::new();

        store
            .expect_events_after_offset()
            .times(2)
            .returning(|| stream::iter(vec![]).boxed());
        store.expect_events_after().times(0);
        store.expect_events_range().times(0);

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            runner.pg_rx.clone(),
            store,
            runner.event_buffer_tx.clone(),
            runner.event_buffer_rx.clone(),
        ));

        sleep(Duration::from_millis(51)).await;

        task.abort();
    }

    #[tokio::test]
    async fn connection_lost_self_healing() {
        let mut runner = TestRunner::new();

        // make sure the task gets a sequence_prior on first run
        let prior_events =
            runner.append_events_only(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);

        let missed_events =
            runner.make_events(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);

        let events = runner.make_events(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);
        let prior_offset_for_missed_events = missed_events.last().unwrap().sequence();

        let mut store = MockStore::new();
        let resume_stream = runner.stream_all();

        // resume call
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| resume_stream);

        // should be called by polling when connection is lost
        store
            .expect_events_after()
            .times(1)
            .with(
                predicate::eq(prior_events.last().unwrap().sequence()),
                predicate::eq(runner.event_buffer_tx.capacity().unwrap()),
            )
            .returning(|_, _| stream::iter(vec![]).boxed());

        // should be called when connection restored and recover missing events
        store
            .expect_events_range()
            .times(1)
            .with(
                predicate::eq(prior_events.last().unwrap().sequence()),
                predicate::eq(events.last().unwrap().sequence()),
            )
            .returning(|_, _| stream::iter(vec![]).boxed());

        let task = tokio::spawn(coordinate_event_channel(
            Duration::from_secs(60),
            runner.pg_rx.clone(),
            store,
            runner.event_buffer_tx.clone(),
            runner.event_buffer_rx.clone(),
        ));

        yield_now().await;

        // drain stream
        runner.event_buffer_rx.drain();

        // simulate network lost
        runner.pg_tx.send(Err(Error::ListenerDisconnected)).unwrap();

        // should be polling at 3 seconds interval and calling expect_events_after
        // wait for one polling cycle to complete
        sleep(Duration::from_millis(3005)).await;

        // simulate a notification with a sequence_prior pointing to some missing events
        let last_event = events.last().unwrap();
        let last_time = last_event.utc().timestamp_nanos();
        let last_seq = last_event.sequence();
        let notification = EventNotification {
            aggregate_id: runner.aggregate_id,
            sequence_start: events[0].sequence(),
            sequence_end: last_seq,
            sequence_prior: Some(prior_offset_for_missed_events),
            time_start: events[0].utc().timestamp_nanos(),
            time_end: last_time,
            total: events.len(),
        };

        runner.pg_tx.send(Ok(notification)).unwrap();

        yield_now().await;

        task.abort();
    }

    //
    // POSTGRES LISTEN NOTIFICATIONS
    //

    #[tokio::test]
    async fn on_notification_streams_range_from_last_known_offset() {
        let poller_frequency = Duration::from_secs(60);
        let mut runner = TestRunner::new();

        let prior_events =
            runner.append_events_only(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);
        let last_prior_event = prior_events.last().unwrap();

        let events = runner.make_events(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);
        let last_event = events.last().unwrap();
        let last_time = last_event.utc().timestamp_nanos();
        let last_seq = last_event.sequence();

        let mut store = MockStore::new();
        let stream_all = runner.stream_all();
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| stream_all);
        store.expect_events_after().times(0);
        store
            .expect_events_range()
            .with(
                predicate::eq(last_prior_event.sequence()),
                predicate::eq(last_seq),
            )
            .times(1)
            .returning(|_, _| stream::iter(vec![]).boxed());

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            runner.pg_rx.clone(),
            store,
            runner.event_buffer_tx.clone(),
            runner.event_buffer_rx.clone(),
        ));

        yield_now().await;

        for event in &events {
            runner.event_buffer_tx.send(Ok(event.clone())).unwrap();
        }

        let notification = EventNotification {
            aggregate_id: runner.aggregate_id,
            sequence_start: events[0].sequence(),
            sequence_end: last_seq,
            sequence_prior: Some(last_prior_event.sequence()),
            time_start: events[0].utc().timestamp_nanos(),
            time_end: last_time,
            total: events.len(),
        };

        runner.pg_tx.send(Ok(notification)).unwrap();

        yield_now().await;

        task.abort();
    }

    #[tokio::test]
    async fn on_notification_streams_min_range_if_no_prior_offset() {
        let poller_frequency = Duration::from_secs(60);
        let mut runner = TestRunner::new();

        let events = runner.make_events(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);
        let last_event = events.last().unwrap();
        let last_time = last_event.utc().timestamp_nanos();
        let last_seq = last_event.sequence();

        let mut store = MockStore::new();
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| stream::iter(vec![]).boxed());
        store.expect_events_after().times(0);
        store
            .expect_events_range()
            .with(predicate::lt(events[0].sequence()), predicate::eq(last_seq))
            .times(1)
            .returning(|_, _| stream::iter(vec![]).boxed());

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            runner.pg_rx.clone(),
            store,
            runner.event_buffer_tx.clone(),
            runner.event_buffer_rx.clone(),
        ));

        yield_now().await;

        for event in &events {
            runner.event_buffer_tx.send(Ok(event.clone())).unwrap();
        }

        let notification = EventNotification {
            aggregate_id: runner.aggregate_id,
            sequence_start: events[0].sequence(),
            sequence_end: last_seq,
            sequence_prior: None,
            time_start: events[0].utc().timestamp_nanos(),
            time_end: last_time,
            total: events.len(),
        };

        runner.pg_tx.send(Ok(notification)).unwrap();

        yield_now().await;

        task.abort();
    }

    #[tokio::test]
    async fn recovers_from_missing_notifications_after_prior_events() {
        let poller_frequency = Duration::from_secs(60);
        let mut runner = TestRunner::new();

        // pre-load events so that there is an offset
        // This is the offset that the task's state will have when notified events arrive
        let prior_events =
            runner.append_events_only(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);

        let offline_events =
            runner.make_events(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);

        let notified_events =
            runner.make_events(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);
        let last_event = notified_events.last().unwrap();
        let last_time = last_event.utc().timestamp_nanos();
        let last_seq = last_event.sequence();

        let mut store = MockStore::new();
        let stream_all = runner.stream_all();
        let events_range = runner.stream_all();
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| stream_all);
        store.expect_events_after().times(0);
        store
            .expect_events_range()
            .with(
                predicate::eq(prior_events.last().unwrap().sequence()),
                predicate::eq(last_seq),
            )
            .times(1)
            .return_once(|_, _| events_range);

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            runner.pg_rx.clone(),
            store,
            runner.event_buffer_tx.clone(),
            runner.event_buffer_rx.clone(),
        ));

        yield_now().await;

        // simulate postgres connection is lost and no notification is received
        for event in &offline_events {
            let offset = event.sequence();
            runner.event_buffer_tx.send(Ok(event.clone())).unwrap();
            runner.last_offset = Some(offset);
        }

        // simulate new events are appended and connection is back on
        for event in &notified_events {
            runner.event_buffer_tx.send(Ok(event.clone())).unwrap();
        }

        let notification = EventNotification {
            aggregate_id: runner.aggregate_id,
            sequence_start: notified_events[0].sequence(),
            sequence_end: last_seq,
            sequence_prior: Some(offline_events.last().unwrap().sequence()),
            time_start: notified_events[0].utc().timestamp_nanos(),
            time_end: last_time,
            total: notified_events.len(),
        };

        runner.pg_tx.send(Ok(notification)).unwrap();

        yield_now().await;

        task.abort();
    }

    #[tokio::test]
    async fn recovers_from_missing_notifications_with_no_prior_events() {
        let poller_frequency = Duration::from_secs(60);
        let mut runner = TestRunner::new();

        let notified_events =
            runner.make_events(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);
        let last_event = notified_events.last().unwrap();
        let last_time = last_event.utc().timestamp_nanos();
        let last_seq = last_event.sequence();

        let mut store = MockStore::new();
        let stream_all = runner.stream_all();
        let events_range = runner.stream_all();
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| stream_all);
        store.expect_events_after().times(0);
        store
            .expect_events_range()
            .with(predicate::eq(std::i64::MIN), predicate::eq(last_seq))
            .times(1)
            .return_once(|_, _| events_range);

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            runner.pg_rx.clone(),
            store,
            runner.event_buffer_tx.clone(),
            runner.event_buffer_rx.clone(),
        ));

        yield_now().await;

        // simulate postgres connection is lost and no notification is received
        let offline_events =
            runner.append_events_only(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);

        // simulate new events are appended and connection is back on
        for event in &notified_events {
            runner.event_buffer_tx.send(Ok(event.clone())).unwrap();
        }

        let notification = EventNotification {
            aggregate_id: runner.aggregate_id,
            sequence_start: notified_events[0].sequence(),
            sequence_end: last_seq,
            sequence_prior: Some(offline_events.last().unwrap().sequence()),
            time_start: notified_events[0].utc().timestamp_nanos(),
            time_end: last_time,
            total: notified_events.len(),
        };

        runner.pg_tx.send(Ok(notification)).unwrap();

        yield_now().await;

        task.abort();
    }

    #[tokio::test]
    async fn skips_invalid_notifications() {
        let poller_frequency = Duration::from_secs(60);
        let mut runner = TestRunner::new();

        let prior_events =
            runner.append_events_only(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);
        let sequence_prior = Some(prior_events[1].sequence());

        let mut store = MockStore::new();
        let stream_all = runner.stream_all();
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| stream_all);
        store.expect_events_after().times(0);
        store.expect_events_range().times(0);

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            runner.pg_rx.clone(),
            store,
            runner.event_buffer_tx.clone(),
            runner.event_buffer_rx.clone(),
        ));

        yield_now().await;

        let events =
            runner.append_events_only(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);

        //
        // Generate a series of invalid notifications for the events just published
        //

        // Simulate a notification that references the second event but should include first too
        let partially_late_notification = EventNotification {
            aggregate_id: runner.aggregate_id,
            sequence_start: prior_events[1].sequence(),
            sequence_end: prior_events[1].sequence(),
            sequence_prior: Some(prior_events[0].sequence()),
            time_start: events[1].utc().timestamp_nanos(),
            time_end: events[1].utc().timestamp_nanos(),
            total: 1,
        };

        let seq_end_after_start = EventNotification {
            aggregate_id: runner.aggregate_id,
            sequence_start: events[1].sequence(),
            sequence_end: events[0].sequence(),
            sequence_prior,
            time_start: events[0].utc().timestamp_nanos(),
            time_end: events[1].utc().timestamp_nanos(),
            total: 2,
        };

        let no_prior_offset = EventNotification {
            sequence_start: events[0].sequence(),
            sequence_end: events[1].sequence(),
            aggregate_id: runner.aggregate_id,
            sequence_prior: None,
            time_start: events[0].utc().timestamp_nanos(),
            time_end: events[1].utc().timestamp_nanos(),
            total: 2,
        };

        runner.pg_tx.send(Ok(partially_late_notification)).unwrap();
        runner.pg_tx.send(Ok(seq_end_after_start)).unwrap();
        runner.pg_tx.send(Ok(no_prior_offset)).unwrap();

        yield_now().await;

        task.abort();
    }

    #[tokio::test]
    async fn skips_late_notifications() {
        let poller_frequency = Duration::from_secs(60);
        let mut runner = TestRunner::new();

        let prior_events =
            runner.append_events_only(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2]);

        let mut store = MockStore::new();
        let stream_all = runner.stream_all();
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| stream_all);
        store.expect_events_after().times(0);
        store.expect_events_range().times(0);

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            runner.pg_rx.clone(),
            store,
            runner.event_buffer_tx.clone(),
            runner.event_buffer_rx.clone(),
        ));

        yield_now().await;

        // The notification for the first events arrives late
        let late_notification = EventNotification {
            sequence_start: prior_events[0].sequence(),
            sequence_end: prior_events[1].sequence(),
            aggregate_id: runner.aggregate_id,
            sequence_prior: None,
            time_start: prior_events[0].utc().timestamp_nanos(),
            time_end: prior_events[1].utc().timestamp_nanos(),
            total: 2,
        };

        runner.pg_tx.send(Ok(late_notification)).unwrap();

        yield_now().await;

        task.abort();
    }

    // This test simulates a race condition where the Subscription
    // 1. Subscribes to postgres notifications
    // 2. An event is published right after and before 3
    // 3. The process calls `events_after_offset` to catch up with events
    // 4. The process gets the event published in (2)
    // 5. When done, it will see the notification for (2) even though the event is already published
    #[tokio::test]
    async fn it_is_no_op_given_a_notification_for_events_which_were_processed_during_bootstrap() {
        let poller_frequency = Duration::from_secs(60);
        let mut runner = TestRunner::new();

        // Instead of just storing "prior_events" for the stream, we also create a notification
        runner
            .append_events_and_notify(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2])
            .await;

        let mut store = MockStore::new();
        let stream_all = runner.stream_all();

        // when process begins and catches up with events
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| stream_all);
        // then it will get a notification as soon as catching up finishes
        // no further operation should be made
        store.expect_events_after().times(0);
        store.expect_events_range().times(0);

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            runner.pg_rx.clone(),
            store,
            runner.event_buffer_tx.clone(),
            runner.event_buffer_rx.clone(),
        ));

        yield_now().await;

        task.abort();
    }

    //
    // BACKOFF
    //

    #[tokio::test]
    async fn should_fetch_according_to_buffer_capacity() {
        let capacity = 10;
        let poller_frequency = Duration::from_secs(60);
        let (buffer_tx, buffer_rx) = flume::bounded::<Result<Persisted<StoreEvents>>>(capacity);
        let mut runner = TestRunner::new();

        let mut store = MockStore::new();
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| stream::iter(vec![]).boxed());
        let events_range_rx = runner.event_buffer_rx.clone();
        store.expect_events_after().times(0);
        store
            .expect_events_range()
            .times(3)
            .returning(move |_, end| {
                let mut buffer = vec![];
                let rx = events_range_rx.clone();

                while let Ok(event) = rx.recv().unwrap() {
                    let is_last = event.sequence() == end;
                    buffer.push(Ok(event));
                    if is_last {
                        break;
                    }
                }
                stream::iter(buffer).boxed()
            });

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            runner.pg_rx.clone(),
            store,
            buffer_tx.clone(),
            buffer_rx.clone(),
        ));

        yield_now().await;

        //
        // Append 11 events
        //
        let inputs = (0..capacity + 1)
            .map(|n| {
                if n % 2 == 0 {
                    StoreEvents::SomeEvent1
                } else {
                    StoreEvents::SomeEvent2
                }
            })
            .collect::<Vec<_>>();
        let mut appended = runner.append_events_and_notify(inputs).await;

        yield_now().await;

        assert!(
            runner.pg_rx.is_empty(),
            "First pg notification should be in flight"
        );
        assert!(
            buffer_rx.is_full(),
            "Subscription buffer should be full before polling any events"
        );

        //
        // Append 2 more events
        //
        appended.extend(
            runner
                .append_events_and_notify(vec![StoreEvents::SomeEvent1])
                .await,
        );
        appended.extend(
            runner
                .append_events_and_notify(vec![StoreEvents::SomeEvent2])
                .await,
        );
        yield_now().await;

        // Events: 10 events in consumer buffer + 1 message in flight (awaiting for capacity) + last 2 messages
        assert!(
            buffer_rx.is_full(),
            "Subscription buffer should remain full"
        );
        assert_eq!(
            runner.event_buffer_rx.len(),
            2,
            "First operation should not be finished yet"
        );
        assert_eq!(
            runner.pg_rx.len(),
            2,
            "Last two pg notification should be buffered"
        );

        //
        // Consume 1 events
        //
        buffer_rx.recv().unwrap().unwrap();

        // allow for congestion to ease with random exponential backoff
        sleep(Duration::from_millis(1500)).await;

        assert!(buffer_rx.is_full(), "Consumer buffer should fill up again");
        assert_eq!(
            runner.pg_rx.len(),
            1,
            "Congestion in notifications buffer should be easing"
        );

        //
        // Consume 3 more events
        //
        for _ in 0..3 {
            buffer_rx.recv().unwrap().unwrap();
        }

        sleep(Duration::from_millis(1500)).await;

        assert_eq!(
            runner.pg_rx.len(),
            0,
            "Shouldn't be more pg notifications in the buffer"
        );
        // messages: 9 in buffer + 4 consumed = 13
        assert_eq!(buffer_rx.len(), 9);

        task.abort();
    }

    #[tokio::test]
    async fn should_use_exponential_backoff_if_db_server_is_overloaded() {
        let poller_frequency = Duration::from_secs(60);
        let mut runner = TestRunner::new();

        let mut store = MockStore::new();
        store
            .expect_events_after_offset()
            .times(1)
            .return_once(|| stream::iter(vec![]).boxed());
        store.expect_events_after().times(0);

        // first time it errors
        store.expect_events_range().times(1).return_once(|_, _| {
            stream::iter(vec![Err(Error::Sqlx(sqlx::Error::PoolTimedOut))]).boxed()
        });
        let events_range = runner.stream_all();
        // then it works
        store
            .expect_events_range()
            .times(1)
            .return_once(|_, _| events_range);

        let task = tokio::spawn(coordinate_event_channel(
            poller_frequency,
            runner.pg_rx.clone(),
            store,
            runner.event_buffer_tx.clone(),
            runner.event_buffer_rx.clone(),
        ));

        yield_now().await;

        let appended = runner
            .append_events_and_notify(vec![StoreEvents::SomeEvent1, StoreEvents::SomeEvent2])
            .await;

        // wait for the exponential backoff
        sleep(Duration::from_millis(1500)).await;

        let events = runner
            .event_buffer_rx
            .drain()
            .map(|x| x.unwrap())
            .collect::<Vec<_>>();

        assert_eq!(events, appended);

        task.abort();
    }
}
