//! Foundation traits for creating Domain abstractions
//! using [the `Aggregate` pattern](https://martinfowler.com/bliki/DDD_Aggregate.html).

use std::fmt::Debug;
use std::marker::Sized;

use async_trait::async_trait;

use chrono::{DateTime, Utc};
use uuid::Uuid;

#[derive(Debug)]
pub struct UncommittedEvent<T> {
    pub utc: DateTime<Utc>,
    pub data: T,
}

impl<T> From<T> for UncommittedEvent<T> {
    fn from(data: T) -> Self {
        Self {
            utc: Utc::now(),
            data,
        }
    }
}

/// An [`Aggregate`] manages a domain entity [`State`](Aggregate::State), acting
/// as a _transaction boundary_.
///
/// It allows **state mutations** through the use of
/// [`Command`](Aggregate::Command)s, which the Aggregate instance handles and
/// emits a number of Domain [`Event`](Aggregate::Event)s.
#[async_trait]
pub trait Aggregate {
    /// State of the Aggregate: this should represent the Domain Entity data
    /// structure.
    type State;

    /// Represents a specific, domain-related change to the Aggregate
    /// [`State`](Aggregate::State).
    type Event;

    /// Possible failures while [`apply`](Aggregate::apply)ing
    /// [`Event`](Aggregate::Event)s or handling
    /// [`Command`](Aggregate::Command)s.
    type Error: std::fmt::Debug + std::error::Error + Send + Sync + 'static;

    /// Commands are all the possible operations available on an Aggregate.
    /// Use Commands to model business use-cases or [`State`](Aggregate::State)
    /// mutations.
    type Command;

    /// Initialise state of root aggregate
    fn apply_first(
        id: Uuid,
        event: &Self::Event,
        utc: DateTime<Utc>,
    ) -> Result<Self::State, Self::Error>;

    /// Apply Self::Event to the current state
    fn apply_next(
        state: &mut Self::State,
        event: &Self::Event,
        utc: DateTime<Utc>,
    ) -> Result<(), Self::Error>;

    /// Transform the constructor Command into Vec<Self::Event>
    fn handle_first(&self, command: Self::Command) -> Result<Vec<Self::Event>, Self::Error>;

    /// Transform a Command into Vec<Self::Event>
    fn handle_next(
        &self,
        state: &Self::State,
        command: Self::Command,
    ) -> Result<Vec<Self::Event>, Self::Error>;

    fn root(&self, id: Uuid) -> AggregateRoot<Self>
    where
        Self: Sized,
    {
        AggregateRoot {
            id,
            last_commit_utc: None,
            state_utc: None,
            state: None,
            aggregate: self,
            uncommitted_events: Default::default(),
        }
    }

    fn root_with_state(
        &self,
        id: Uuid,
        utc: DateTime<Utc>,
        state: Self::State,
    ) -> AggregateRoot<Self>
    where
        Self: Sized,
    {
        AggregateRoot {
            id,
            last_commit_utc: Some(utc),
            state_utc: Some(utc),
            state: Some(state),
            aggregate: self,
            uncommitted_events: Default::default(),
        }
    }
}

#[derive(Debug)]
pub struct AggregateRoot<'a, T>
where
    T: Aggregate + 'static,
{
    id: Uuid,
    last_commit_utc: Option<DateTime<Utc>>,
    state_utc: Option<DateTime<Utc>>,
    state: Option<T::State>,
    aggregate: &'a T,
    uncommitted_events: Vec<UncommittedEvent<T::Event>>,
}

impl<'a, T> PartialEq for AggregateRoot<'a, T>
where
    T: Aggregate,
{
    #[inline]
    fn eq(&self, other: &Self) -> bool {
        self.id() == other.id()
    }
}

impl<'a, T> AggregateRoot<'a, T>
where
    T: Aggregate,
{
    pub fn id(&self) -> Uuid {
        self.id
    }

    pub fn state(&self) -> Option<(&T::State, DateTime<Utc>)> {
        match (&self.state, self.state_utc) {
            (Some(state), Some(utc)) => Some((state, utc)),
            _ => None,
        }
    }

    pub fn uncommitted_events(&self) -> &Vec<UncommittedEvent<T::Event>> {
        self.uncommitted_events.as_ref()
    }

    pub fn last_commit_utc(&self) -> Option<DateTime<Utc>> {
        self.last_commit_utc
    }

    pub async fn handle(
        &mut self,
        command: T::Command,
    ) -> Result<&mut AggregateRoot<'a, T>, T::Error> {
        let next_events = match self.state.as_ref() {
            Some(state) => self
                .aggregate
                .handle_next(state, command)?
                .into_iter()
                .map(UncommittedEvent::from)
                .collect::<Vec<_>>(),
            None => self
                .aggregate
                .handle_first(command)?
                .into_iter()
                .map(UncommittedEvent::from)
                .collect::<Vec<_>>(),
        };

        for event in &next_events {
            // update state
            let data = &event.data;
            let utc = event.utc;

            if let Some(ref mut state) = self.state {
                T::apply_next(state, data, utc)?;
            } else {
                let first_state = T::apply_first(self.id, data, utc)?;
                self.state = Some(first_state);
            }
        }

        self.state_utc = next_events.last().map(|e| e.utc);
        self.uncommitted_events.extend(next_events);

        Ok(self)
    }

    pub(crate) fn commit(&mut self) {
        self.uncommitted_events.clear();
        self.last_commit_utc = self.state_utc;
    }
}
