/**
 *
 * EXTENSIONS
 *
 */

CREATE EXTENSION IF NOT EXISTS timescaledb;

CREATE OR REPLACE FUNCTION unix_now()
returns BIGINT
LANGUAGE SQL STABLE as $$
    SELECT extract(epoch from now())::BIGINT
$$;

/**
 *
 * TABLES
 *
 */

CREATE TABLE aggregate_type (
    id              SERIAL      PRIMARY KEY,
    name            TEXT        NOT NULL UNIQUE
);

CREATE TABLE event_name (
    id                  SERIAL    PRIMARY KEY,
    name                TEXT      NOT NULL,
    aggregate_type_id   INTEGER   NOT NULL,

    UNIQUE (aggregate_type_id, name),
    -- Remove all event_names in case the aggregate type is deleted.
    FOREIGN KEY (aggregate_type_id) REFERENCES aggregate_type ON DELETE CASCADE
);

CREATE TABLE event (
    time                BIGINT      NOT NULL,
    sequence            BIGSERIAL,    
    aggregate_id        UUID        NOT NULL,
    aggregate_type_id   INTEGER     NOT NULL REFERENCES aggregate_type,
    name_id             INTEGER     NOT NULL REFERENCES event_name,
    payload             JSONB,

    PRIMARY KEY (aggregate_type_id, aggregate_id, time)
);

-- chunk_time_interval is default of one week for BIGINT
SELECT create_hypertable('event', 'time', 'aggregate_type_id', 5, chunk_time_interval => 604800000000000, create_default_indexes => false);
SELECT set_integer_now_func('event', 'unix_now');
-- Index used by subscriptions
CREATE INDEX ON event (aggregate_type_id, sequence);

CREATE TABLE aggregate_subscription (
    name                 TEXT           PRIMARY KEY,
    aggregate_type_id    INTEGER        NOT NULL,
    "offset"             BIGINT,

    -- Remove all subscriptions in case the aggregate type is deleted.
    FOREIGN KEY (aggregate_type_id) REFERENCES aggregate_type ON DELETE CASCADE

    -- foreign keys referencing hypertables not supported by timescaledb at the moment
    -- see https://github.com/timescale/timescaledb/issues/498
    -- FOREIGN KEY (offset) REFERENCES event(time)
);


/**
 *
 * PROCEDURES
 *
 */

CREATE TYPE append_event_data AS (
    data            JSONB,
    name_id         INTEGER,
    time            BIGINT
);

CREATE OR REPLACE FUNCTION append_to_store(
    _aggregate_type_id     INTEGER,
    _aggregate_id          UUID,
    event_data_array       append_event_data[],
    orderly                BOOLEAN,
    client_last_known_time BIGINT,
    OUT last_event_time    BIGINT
)
AS $$
DECLARE
    event_data                  append_event_data;
    event_name_id               INTEGER;
    event_payload               JSONB;    
    event_time                  BIGINT;
    first_event_time            BIGINT;    
    prior_event                 RECORD;
    sequence_prior              BIGINT;
    sequence_end                BIGINT;
    sequence_start              BIGINT;
BEGIN    

    -- Retrieve the last event for the specified aggregate id.
    SELECT time, sequence
    INTO prior_event
    FROM event
    WHERE
        aggregate_type_id = _aggregate_type_id AND
        aggregate_id = _aggregate_id
    ORDER BY time DESC
    LIMIT 1
    FOR UPDATE; -- if row exists, get an exclussive lock which implicitly blocks more events being appended to the same aggregate

    -- TODO: is this lock necessary? can a clash occur if random aggregate_id is generated in each distributed client?
    -- IF NOT FOUND THEN 
    --     PERFORM pg_advisory_xact_lock(1);
    -- END IF;

    sequence_prior = prior_event.sequence;
    last_event_time = prior_event.time;
    
    -- Perform optimistic concurrency check.
    IF orderly AND last_event_time IS DISTINCT FROM client_last_known_time THEN
        RAISE EXCEPTION 'invalid aggregate time provided: %, expected: %', client_last_known_time, last_event_time;
    END IF;

    FOREACH event_data IN ARRAY event_data_array
    LOOP
        event_name_id = event_data.name_id;
        event_payload= event_data.data;
        event_time = event_data.time;

        -- check that events happened after last known offset
        IF last_event_time > event_time THEN
            IF orderly THEN
                RAISE EXCEPTION 'events must be in order. Got: %, expected to be after: %', event_time, last_event_time;
            ELSE
                -- use last offset to reorder event
                -- NOTE: using unix_now() with subsec microsecond precission is not enough
                --       It could create collision if there are more events in loop, as next could get the same time
                event_time = last_event_time + 1;
            END IF;
        END IF;

        INSERT INTO event
            (time, aggregate_id, aggregate_type_id, name_id, payload)
        VALUES
            (event_time, _aggregate_id, _aggregate_type_id, event_name_id, event_payload)
        RETURNING sequence INTO sequence_end;

        IF sequence_start IS NULL THEN
            sequence_start = sequence_end;
        END IF;

        last_event_time = event_time;

    END LOOP;

    IF cardinality(event_data_array) > 0 THEN
        SELECT time
        INTO first_event_time
        FROM unnest(event_data_array)
        LIMIT 1;

        -- Wake up listeners of _aggregate_type_id channel
        PERFORM pg_notify(_aggregate_type_id::TEXT, ''
            || '{'
            || '"aggregate_id":"' || _aggregate_id                          || '",'
            || '"sequence_prior":'|| coalesce(sequence_prior::TEXT, 'null') || ','
            || '"sequence_start":'|| sequence_start                         || ','    
            || '"sequence_end":'  || sequence_end                           || ',' 
            || '"time_start":'    || first_event_time                       || ',' 
            || '"time_end":'      || last_event_time                        || ','    
            || '"total":'         || cardinality(event_data_array)                                            
            || '}');
    END IF;
    
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION aggregate_type_id(
    _name TEXT,
    OUT _id INT)
AS $$
BEGIN
LOOP
    SELECT id
    FROM aggregate_type
    WHERE name = _name
    INTO _id;

    EXIT WHEN FOUND;

    INSERT INTO aggregate_type (name)
    VALUES (_name)
    ON CONFLICT (name) DO NOTHING
    RETURNING id
    INTO _id;

   EXIT WHEN FOUND;
END LOOP;
END
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION event_name_id(
    _aggregate_type_id INTEGER,
    _name TEXT,
    OUT _id INT
) AS $$
BEGIN
LOOP
    SELECT id
    FROM event_name
    WHERE 
        aggregate_type_id = _aggregate_type_id AND
        name = _name
    INTO _id;

    EXIT WHEN FOUND;

    INSERT INTO event_name (aggregate_type_id, name)
    VALUES (_aggregate_type_id, _name)
    ON CONFLICT (aggregate_type_id, name) DO NOTHING
    RETURNING id
    INTO _id;

    EXIT WHEN FOUND;
END LOOP;
END
$$ LANGUAGE plpgsql;


CREATE OR REPLACE FUNCTION subscription_offset(
    _name               TEXT,
    _aggregate_type_id  INTEGER,
    OUT _offset         BIGINT
) AS $$
BEGIN
LOOP
    
    SELECT "offset"
    FROM aggregate_subscription
    WHERE name = _name
    INTO _offset;

    EXIT WHEN FOUND;

    INSERT INTO aggregate_subscription (name, aggregate_type_id)
    VALUES (_name, _aggregate_type_id)
    ON CONFLICT (name) DO NOTHING
    RETURNING "offset"
    INTO _offset;

    EXIT WHEN FOUND;
    
END LOOP;
END
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION try_subscription_checkout(
    _name               TEXT,
    _offset             BIGINT
)
RETURNS void
AS $$
DECLARE
    _subscription               RECORD;
    _last_committed_offset      BIGINT;
    _aggregate_type_id          INT;
    _anterior_offset            BIGINT;
    _event                      RECORD;
BEGIN
    
    SELECT "offset", aggregate_type_id
    FROM aggregate_subscription
    INTO _subscription
    WHERE name = _name
    FOR UPDATE; -- lock

    _last_committed_offset = _subscription.offset;
    _aggregate_type_id= _subscription.aggregate_type_id;

    IF _last_committed_offset IS NOT NULL AND _offset <= _last_committed_offset THEN
        RAISE EXCEPTION 'checkpoint behind current offset. Received %, expected higher than %', _offset, _last_committed_offset;
    END IF;

    -- Get event for the offset and the previous one
    CREATE TEMP TABLE _events ON COMMIT DROP AS
    SELECT sequence
    FROM event
    WHERE
        aggregate_type_id = _aggregate_type_id AND
        sequence <= _offset
    ORDER BY sequence DESC
    LIMIT 2;

    -- Subscription table can't reference "event" hypertable
    -- Perform faux FOREIGN key check here
    SELECT sequence
    INTO _event
    FROM _events e
    WHERE _offset = e.sequence;

    IF NOT FOUND THEN
        RAISE EXCEPTION 'update or delete on table aggregate_subscription violates foreign key constraint. No event found for offset: %', _offset;
    END IF;

    -- Do a constraint check for order
    SELECT sequence
    INTO _anterior_offset
    FROM _events
    WHERE sequence < _offset;

    IF _last_committed_offset IS DISTINCT FROM _anterior_offset THEN
        RAISE EXCEPTION 'checkpoint out of order. Offset % does not follow event with timestamp %', _offset, _anterior_offset;
    END IF;

    UPDATE aggregate_subscription
    SET "offset" = _offset
    WHERE "name" = _name;
    
END;
$$ LANGUAGE plpgsql;