use std::{borrow::Cow, convert::TryFrom};

use chrono::{DateTime, TimeZone, Utc};
use serde::{Deserialize, Serialize};
use serde_json::Value as Json;
use sqlx::FromRow;
use uuid::Uuid;

/// An [`Event`](EventStore::Event) wrapper for events that have been
/// successfully committed to the [`EventStore`].
///
/// [`EventStream`]s are composed of these events.
#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub struct Persisted<T> {
    aggregate_id: Uuid,
    utc: DateTime<Utc>,
    sequence: i64,
    data: T,
}

impl<T> Persisted<T> {
    /// Returns the event sequence number.
    pub fn sequence(&self) -> i64 {
        self.sequence
    }

    /// Returns the aggregate id.
    pub fn aggregate_id(&self) -> Uuid {
        self.aggregate_id
    }

    pub fn utc(&self) -> DateTime<Utc> {
        self.utc
    }

    pub fn data(&self) -> &T {
        &self.data
    }

    /// Unwraps the inner [`Event`](EventStore::Event) from the `Persisted`
    /// wrapper.
    pub fn into_data(self) -> T {
        self.data
    }
}

pub trait TimescaleEventPayload {
    fn name(&self) -> Cow<'static, str>;
}

#[derive(Debug, FromRow)]
pub struct EventRow {
    pub aggregate_id: Uuid,
    pub sequence: i64,
    pub name: String,
    pub payload: Option<Json>,
    pub time: i64,
}

impl TimescaleEventPayload for EventRow {
    fn name(&self) -> Cow<'static, str> {
        self.name.clone().into()
    }
}

impl<T> TryFrom<EventRow> for Persisted<T>
where
    for<'de> T: Deserialize<'de>,
{
    type Error = serde_json::error::Error;

    fn try_from(row: EventRow) -> serde_json::error::Result<Self> {
        let aggregate_id = row.aggregate_id;
        let utc = Utc.timestamp_nanos(row.time);
        let sequence = row.sequence;

        let event = Persisted {
            aggregate_id,
            utc,
            sequence,
            data: parse_like_externally_tagged(row.name, row.payload)?,
        };

        Ok(event)
    }
}

// reconstruct externally tagged structs
fn parse_like_externally_tagged<T>(
    name: String,
    payload: Option<Json>,
) -> serde_json::error::Result<T>
where
    for<'de> T: Deserialize<'de>,
{
    let value = match payload {
        None => {
            // serialize event payload into a unit struct
            Json::String(name)
        }
        Some(value) => {
            let mut map = serde_json::Map::new();
            map.insert(name, value);

            Json::Object(map)
        }
    };

    serde_json::from_value(value)
}
