use std::convert::TryFrom;
use std::io::Write;

use clap::ArgMatches;
use chrono::{DateTime, Utc, Duration, Local};
use itertools::Itertools;

use crate::error::{Error, Result};
use crate::database::Database;
use crate::config::Config;
use crate::tabulate::{Tabulate, Col, Align::*};
use crate::formatters::text::format_duration;
use crate::models::Entry;
use crate::old::{entries_or_warning, warn_if_needed};

use super::Command;

#[derive(Default)]
pub struct Args {
    all: bool,
}

impl<'a> TryFrom<&'a ArgMatches<'a>> for Args {
    type Error = Error;

    fn try_from(matches: &ArgMatches) -> Result<Args> {
        Ok(Args {
            all: matches.is_present("all"),
        })
    }
}

pub struct ListCommand {}

impl<'a> Command<'a> for ListCommand {
    type Args = Args;

    fn handle<D, O, E>(args: Args, db: &mut D, out: &mut O, err: &mut E, _config: &Config, now: DateTime<Utc>) -> Result<()>
    where
        D: Database,
        O: Write,
        E: Write,
    {
        let today = now.with_timezone(&Local).date().and_hms(0, 0, 0).with_timezone(&Utc);
        let entries = if args.all {
            db.entries_full(None, None)?
        } else {
            db.entries_all_visible(None, None)?
        };

        let (mut entries, needs_warning) = entries_or_warning(entries, db)?;

        let current = db.current_sheet()?;
        let last = db.last_sheet()?;

        // introducte two fake entries to make both current and last show up
        if let Some(ref current) = current {
            entries.push(Entry {
                id: 1, sheet: current.clone(), start: now, end: Some(now), note: None,
            });
        }

        entries.sort_unstable_by_key(|e| e.sheet.clone());

        let sheets: Vec<_> = entries
            .into_iter()
            .group_by(|e| e.sheet.clone())
            .into_iter()
            .map(|(key, group)| {
                let entries: Vec<_> = group.into_iter().collect();

                (
                    if current.as_ref() == Some(&key) {
                        "*".into()
                    } else if last.as_ref() == Some(&key) {
                        "-".into()
                    } else {
                        "".into()
                    },
                    key,

                    format_duration(
                        now - entries.iter().find(|e| e.end.is_none()).map(|e| e.start).unwrap_or(now)
                    ),

                    format_duration(
                        entries.iter().filter(|e| e.start > today).fold(Duration::seconds(0), |acc, e| {
                            acc + (e.end.unwrap_or(now) - e.start)
                        })
                    ),

                    // total
                    format_duration(entries.into_iter().fold(Duration::seconds(0), |acc, e| {
                        acc + (e.end.unwrap_or(now) - e.start)
                    })),
                )
            })
            .collect();

        let mut tabs = Tabulate::with_columns(vec![
            // indicator of current or prev sheet
            Col::min_width(1).and_alignment(Right),

            // sheet name
            Col::min_width(9).and_alignment(Left),

            // running time
            Col::min_width(9).and_alignment(Right),

            // today
            Col::min_width(9).and_alignment(Right),

            // accumulated
            Col::min_width(12).and_alignment(Right),
        ]);

        tabs.feed(vec!["".into(), "Timesheet".into(), "Running".into(), "Today".into(), "Total Time".into()]);
        tabs.separator(' ');

        for sheet in sheets {
            tabs.feed(vec![sheet.0, sheet.1, sheet.2, sheet.3, sheet.4]);
        }

        out.write_all(tabs.print().as_bytes())?;

        warn_if_needed(err, needs_warning)?;

        Ok(())
    }
}

#[cfg(test)]
mod tests {
    use chrono::{Utc, TimeZone};
    use pretty_assertions::assert_eq;
    use ansi_term::Color::Yellow;

    use crate::database::{SqliteDatabase, Database};
    use crate::test_utils::Ps;

    use super::*;

    #[test]
    fn list_sheets() {
        std::env::set_var("TZ", "CST+6");

        let args = Default::default();
        let mut db = SqliteDatabase::from_memory().unwrap();
        let mut out = Vec::new();
        let mut err = Vec::new();
        let config = Default::default();
        db.init().unwrap();
        db.set_current_sheet("sheet2").unwrap();
        db.set_last_sheet("sheet4").unwrap();

        db.entry_insert(Utc.ymd(2021, 1, 1).and_hms(0, 0, 0), Some(Utc.ymd(2021, 1, 1).and_hms(1, 0, 0)), None, "_archived".into()).unwrap();
        db.entry_insert(Utc.ymd(2021, 1, 1).and_hms(0, 0, 0), Some(Utc.ymd(2021, 1, 1).and_hms(10,13, 55)), None, "sheet1".into()).unwrap();
        db.entry_insert(Utc.ymd(2021, 1, 1).and_hms(0, 0, 0), Some(Utc.ymd(2021, 1, 1).and_hms(7, 39, 18)), None, "sheet3".into()).unwrap();
        db.entry_insert(Utc.ymd(2021, 1, 1).and_hms(12, 0, 0), Some(Utc.ymd(2021, 1, 1).and_hms(13, 52, 45)), None, "sheet3".into()).unwrap();
        db.entry_insert(Utc.ymd(2021, 1, 1).and_hms(12, 0, 0), None, None, "sheet4".into()).unwrap();

        let now = Utc.ymd(2021, 1, 1).and_hms(13, 52, 45);

        ListCommand::handle(args, &mut db, &mut out, &mut err, &config, now).unwrap();

        assert_eq!(Ps(&String::from_utf8_lossy(&out)), Ps("  Timesheet   Running     Today   Total Time

  sheet1      0:00:00   0:00:00     10:13:55
* sheet2      0:00:00   0:00:00      0:00:00
  sheet3      0:00:00   1:52:45      9:32:03
- sheet4      1:52:45   1:52:45      1:52:45
"));

        // now show all the sheets
        let mut out = Vec::new();
        let mut err = Vec::new();
        let args = Args {
            all: true,
        };

        ListCommand::handle(args, &mut db, &mut out, &mut err, &config, now).unwrap();

        assert_eq!(Ps(&String::from_utf8_lossy(&out)), Ps("  Timesheet   Running     Today   Total Time

  _archived   0:00:00   0:00:00      1:00:00
  sheet1      0:00:00   0:00:00     10:13:55
* sheet2      0:00:00   0:00:00      0:00:00
  sheet3      0:00:00   1:52:45      9:32:03
- sheet4      1:52:45   1:52:45      1:52:45
"));
    }

    #[test]
    fn old_database() {
        let args = Default::default();
        let mut db = SqliteDatabase::from_path("assets/test_list_old_database.db").unwrap();
        let mut out = Vec::new();
        let mut err = Vec::new();
        let config = Default::default();

        let now = Local.ymd(2021, 7, 16).and_hms(11, 30, 45);

        ListCommand::handle(args, &mut db, &mut out, &mut err, &config, now.with_timezone(&Utc)).unwrap();

        assert_eq!(Ps(&String::from_utf8_lossy(&out)), Ps("  Timesheet   Running     Today   Total Time

* default     0:10:24   0:10:26      0:10:26
"));

        assert_eq!(
            String::from_utf8_lossy(&err),
            format!("{} You are using the old timetrap format, it is advised that you update your database using t migrate\n", Yellow.bold().paint("[WARNING]")),
        );
    }
}
