use std::convert::TryFrom;
use std::io::Write;

use clap::ArgMatches;
use chrono::{DateTime, Utc, Local};
use regex::Regex;

use crate::error::{Result, Error};
use crate::database::Database;
use crate::formatters::Formatter;
use crate::config::Config;
use crate::timeparse::parse_time;
use crate::regex::parse_regex;

use super::{Command, display::{Sheet, entries_for_display}};

#[derive(Default)]
pub struct Args {
    ids: bool,
    end: Option<DateTime<Utc>>,
    format: Formatter,
    grep: Option<Regex>,
    sheet: Option<Sheet>,
}

impl<'a> TryFrom<&'a ArgMatches<'a>> for Args {
    type Error = Error;

    fn try_from(matches: &'a ArgMatches) -> Result<Args> {
        Ok(Args {
            ids: matches.is_present("ids"),
            end: matches.value_of("end").map(|s| parse_time(s)).transpose()?,
            format: matches.value_of("format").unwrap().parse()?,
            grep: matches.value_of("grep").map(parse_regex).transpose()?,
            sheet: matches.value_of("sheet").map(|s| s.parse()).transpose()?,
        })
    }
}

pub struct TodayCommand { }

impl<'a> Command<'a> for TodayCommand {
    type Args = Args;

    fn handle<D, O, E>(args: Self::Args, db: &mut D, out: &mut O, err: &mut E, _config: &Config, now: DateTime<Utc>) -> Result<()>
    where
        D: Database,
        O: Write,
        E: Write,
    {
        let start = Some(now.with_timezone(&Local).date().and_hms(0, 0, 0).with_timezone(&Utc));

        entries_for_display(start, args.end, args.sheet, db, out, err, args.format, args.ids, args.grep, now)
    }
}
