use std::convert::TryFrom;
use std::io::Write;

use clap::ArgMatches;
use chrono::{DateTime, Utc};

use crate::error::{Error, Result};
use crate::database::Database;
use crate::config::Config;
use crate::interactive::ask;

use super::Command;

#[derive(Debug)]
pub enum Args {
    Id(u64),
    Sheet(String),
}

impl<'a> TryFrom<&'a ArgMatches<'a>> for Args {
    type Error = Error;

    fn try_from(args: &'a ArgMatches) -> Result<Args> {
        Ok(args.value_of("id").map(|v| {
            Args::Id(v.parse().unwrap())
        }).unwrap_or_else(|| {
            Args::Sheet(args.value_of("sheet").unwrap().to_owned())
        }))
    }
}

pub struct KillCommand;

impl<'a> Command<'a> for KillCommand {
    type Args = Args;

    fn handle<D, O, E>(args: Args, db: &mut D, out: &mut O, _err: &mut E, _config: &Config, _now: DateTime<Utc>) -> Result<()>
    where
        D: Database,
        O: Write,
        E: Write,
    {
        match args {
            Args::Id(id) => {
                if let Some(entry) = db.entry_by_id(id)? {
                    if ask(out, &format!("are you sure you want to delete entry {}? ({})", entry.id, entry.note.unwrap_or_else(|| "".into())))? {
                        db.delete_entry_by_id(id)?;
                        writeln!(out, "It's dead")?;
                    } else {
                        writeln!(out, "Don't worry, it's still there")?;
                    }
                } else {
                    writeln!(out, "There's no entry with id {}. Someone found it before we did.", id)?;
                }
            },
            Args::Sheet(sheet) => {
                let n = db.entries_by_sheet(&sheet, None, None)?.len();

                if ask(out, &format!("are you sure you want to delete {} entries on sheet \"{}\"?", n, sheet))? {
                    db.delete_entries_in_sheet(&sheet)?;
                    writeln!(out, "They're gone")?;
                } else {
                    writeln!(out, "Don't worry, they're still there")?;
                }
            }
        }

        Ok(())
    }
}
