use std::convert::TryFrom;
use std::io::Write;

use clap::ArgMatches;
use chrono::{DateTime, Utc, Local, Duration};
use regex::Regex;

use crate::error::{Result, Error};
use crate::database::Database;
use crate::formatters::Formatter;
use crate::config::Config;
use crate::regex::parse_regex;

use super::{Command, display::{Sheet, entries_for_display}};

#[derive(Default)]
pub struct Args {
    ids: bool,
    format: Formatter,
    grep: Option<Regex>,
    sheet: Option<Sheet>,
}

impl<'a> TryFrom<&'a ArgMatches<'a>> for Args {
    type Error = Error;

    fn try_from(matches: &'a ArgMatches) -> Result<Args> {
        Ok(Args {
            ids: matches.is_present("ids"),
            format: matches.value_of("format").unwrap().parse()?,
            grep: matches.value_of("grep").map(parse_regex).transpose()?,
            sheet: matches.value_of("sheet").map(|s| s.parse()).transpose()?,
        })
    }
}

pub struct YesterdayCommand { }

impl<'a> Command<'a> for YesterdayCommand {
    type Args = Args;

    fn handle<D, O, E>(args: Self::Args, db: &mut D, out: &mut O, err: &mut E, _config: &Config, now: DateTime<Utc>) -> Result<()>
    where
        D: Database,
        O: Write,
        E: Write,
    {
        let today = now.with_timezone(&Local).date();
        let start = Some((today - Duration::days(1)).and_hms(0, 0, 0).with_timezone(&Utc));
        let end = Some(today.and_hms(0, 0, 0).with_timezone(&Utc));

        entries_for_display(start, end, args.sheet, db, out, err, args.format, args.ids, args.grep, now)
    }
}

#[cfg(test)]
mod tests {
    use chrono::Duration;
    use pretty_assertions::assert_eq;

    use crate::database::SqliteDatabase;
    use crate::test_utils::Ps;

    use super::*;

    #[test]
    fn returns_yesterday_entries_only() {
        let args = Args {
            format: Formatter::Csv,
            ..Default::default()
        };
        let mut db = SqliteDatabase::from_memory().unwrap();
        let mut out = Vec::new();
        let mut err = Vec::new();
        let config = Default::default();

        db.init().unwrap();

        let two_days_ago = Local::now().date() - Duration::days(2);
        let yesterday = Local::now().date() - Duration::days(1);
        let today = Local::now().date();

        db.entry_insert(two_days_ago.and_hms(1, 2, 3).with_timezone(&Utc), None, None, "default".into()).unwrap();
        db.entry_insert(yesterday.and_hms(1, 2, 3).with_timezone(&Utc), None, Some("This!".into()), "default".into()).unwrap();
        db.entry_insert(today.and_hms(1, 2, 3).with_timezone(&Utc), None, None, "default".into()).unwrap();

        YesterdayCommand::handle(args, &mut db, &mut out, &mut err, &config, Utc::now()).unwrap();

        assert_eq!(Ps(&String::from_utf8_lossy(&out)), Ps(&format!("start,end,note,sheet
{},,This!,default
", yesterday.and_hms(1, 2, 3).with_timezone(&Utc).to_rfc3339_opts(chrono::SecondsFormat::Micros, true))));

        assert_eq!(
            String::from_utf8_lossy(&err),
            String::new(),
        );
    }
}
