use noodles::bgzf;
use noodles::vcf;
use vcf::header::info::Key;
use vcf::record::chromosome::Chromosome;
use vcf::record::filters::Filters;
use vcf::record::info::field::Value;

use structopt::StructOpt;

use std::error::Error;
use std::fs::File;
use std::io::{stdin, stdout, BufReader, Read, Write};

mod cli;

use cli::Opt;

const HEADER_START: [&str; 7] = ["contig", "pos", "id", "ref", "alt", "qual", "filter"];
const VEP_DESC_PREFIX: &str = "Consequence annotations from Ensembl VEP. Format: ";

enum TableKind {
    Cartesian,
    Stacked,
}

fn main() -> Result<(), Box<dyn Error>> {
    let opt = Opt::from_args();

    let kind = match opt.stack {
        true => TableKind::Stacked,
        false => TableKind::Cartesian,
    };

    let input: Box<dyn Read> = match opt.input.clone() {
        Some(path)
	    // gotta be a better way; ==Some("gz") doesn't work
            if opt.bgzip || (path.extension().is_some() && path.extension().unwrap() == "gz") =>
        {
            Box::new(bgzf::Reader::new(File::open(path)?))
        }
        Some(path) => Box::new(File::open(path)?),
        None if opt.bgzip => Box::new(bgzf::Reader::new(stdin())),
        None => Box::new(stdin()),
    };

    let output: Box<dyn Write> = match opt.output.clone() {
        Some(path) => Box::new(File::create(path)?),
        None => Box::new(stdout()),
    };

    convert_records(input, output, kind, opt)?;

    Ok(())
}

fn convert_records(
    input: Box<dyn Read>,
    writer: Box<dyn Write>,
    kind: TableKind,
    opt: Opt,
) -> Result<(), Box<dyn Error>> {
    let mut reader = vcf::Reader::new(BufReader::new(input));

    let writer = match opt.csv {
        true => csv::WriterBuilder::new()
            .delimiter(b',')
            .from_writer(writer),
        false => csv::WriterBuilder::new()
            .delimiter(b'\t')
            .from_writer(writer),
    };

    let header: vcf::Header = reader.read_header()?.parse()?;

    let info_prefix = match opt.info_prefix {
        Some(p) => p,
        None => "info_".to_owned(),
    };

    match kind {
        TableKind::Cartesian => {
            let sample_delim = match opt.format_delim {
                Some(d) => d,
                None => "_".to_string(),
            };
            convert_records_cartesian(
                reader,
                writer,
                header,
                sample_delim,
                info_prefix,
                opt.vep_fields,
            )?
        }
        TableKind::Stacked => {
            convert_records_stacked(reader, writer, header, info_prefix, opt.vep_fields)?
        }
    }
    Ok(())
}

// unneccesary code duplication between this + _cartesian
fn convert_records_stacked(
    mut reader: vcf::Reader<BufReader<impl Read>>,
    mut writer: csv::Writer<impl Write>,
    header: vcf::Header,
    info_prefix: String,
    split_csq: bool,
) -> Result<(), Box<dyn Error>> {
    let fmts = header.formats();
    let infos = header.infos();
    let samples = header.sample_names();

    let mut info_header = Vec::new();
    let mut number_csq = 0;

    for key in infos.keys() {
        if split_csq {
            match key {
                Key::Other(name, ..) if name == "CSQ" => {
                    let (_, fields) = infos
                        .get(key)
                        .unwrap()
                        .description()
                        .split_at(VEP_DESC_PREFIX.len());
                    let mut csq_fields: Vec<String> =
                        fields.split('|').map(ToString::to_string).collect();
                    number_csq = csq_fields.len() + 1;
                    info_header.append(&mut csq_fields);
                    info_header.push("CSQ_other_transcripts".to_owned());
                }
                _ => info_header.push(format!("{}{}", info_prefix, key)),
            }
        } else {
            info_header.push(format!("{}{}", info_prefix, key));
        }
    }

    let mut out_header: Vec<String> = HEADER_START.map(ToString::to_string).to_vec();
    out_header.append(&mut info_header);

    let mut fmt_header: Vec<String> = fmts
        .keys()
        .map(ToString::to_string)
        .collect::<Vec<String>>();

    fmt_header.push("sample".to_owned());

    out_header.append(&mut fmt_header);
    writer.write_record(&out_header)?;

    for result in reader.records(&header) {
        let record = result?;

        let mut locus_fields = main_fields_from_record(&record);

        push_info_fields(infos, record.info(), &mut locus_fields, number_csq);

        let sample_fmts = record.genotypes();

        for (sample_fmt, sample) in sample_fmts.iter().zip(samples) {
            let mut fields = locus_fields.clone();
            for key in fmts.keys() {
                if let Some(field) = sample_fmt.get(key) {
                    fields.push(field.to_string());
                } else {
                    fields.push("NA".to_owned());
                }
            }
            fields.push(sample.clone());
            writer.write_record(&fields)?;
        }
    }
    Ok(())
}

fn convert_records_cartesian(
    mut reader: vcf::Reader<BufReader<impl Read>>,
    mut writer: csv::Writer<impl Write>,
    header: vcf::Header,
    sample_delim: String,
    info_prefix: String,
    split_csq: bool,
) -> Result<(), Box<dyn Error>> {
    let fmts = header.formats();
    let infos = header.infos();
    let samples = header.sample_names();

    let mut info_header = Vec::new();
    let mut number_csq = 0;

    for key in infos.keys() {
        if split_csq {
            match key {
                Key::Other(name, ..) if name == "CSQ" => {
                    let (_, fields) = infos
                        .get(key)
                        .unwrap()
                        .description()
                        .split_at(VEP_DESC_PREFIX.len());
                    let mut csq_fields: Vec<String> =
                        fields.split('|').map(ToString::to_string).collect();
                    number_csq = csq_fields.len() + 1;
                    info_header.append(&mut csq_fields);
                    info_header.push("CSQ_other_transcripts".to_owned());
                }
                _ => info_header.push(format!("{}{}", info_prefix, key)),
            }
        } else {
            info_header.push(format!("{}{}", info_prefix, key));
        }
    }

    let mut out_header: Vec<String> = HEADER_START.map(ToString::to_string).to_vec();
    out_header.append(&mut info_header);
    let mut fmt_header: Vec<String> = samples
        .iter()
        .flat_map(|sample| {
            fmts.keys()
                .map(|field| format!("{}{}{}", sample, sample_delim, field))
                .collect::<Vec<String>>()
        })
        .collect();

    out_header.append(&mut fmt_header);
    writer.write_record(&out_header)?;

    for result in reader.records(&header) {
        let record = result?;

        let mut fields = main_fields_from_record(&record);

        push_info_fields(infos, record.info(), &mut fields, number_csq);

        let sample_fmts = record.genotypes();

        for sample_fmt in sample_fmts.iter() {
            for key in fmts.keys() {
                if let Some(field) = sample_fmt.get(key) {
                    fields.push(field.to_string());
                } else {
                    fields.push("NA".to_owned());
                }
            }
        }

        writer.write_record(&fields)?;
    }
    Ok(())
}

fn push_info_fields(
    infos: &vcf::header::Infos,
    info: &vcf::record::Info,
    fields: &mut Vec<String>,
    split_csq: usize,
) {
    for info_field in infos.keys() {
        match info_field {
            // CSQ field and we want to split it
            Key::Other(ref name, ..) if split_csq != 0 && name == "CSQ" => {
                let field = info
                    .get(info_field)
                    .expect("No CSQ field!")
                    .value()
                    .expect("No CSQ value!");
                let mut subfields: Vec<String> = match field {
                    Value::String(s) => s.splitn(split_csq, '|').map(ToString::to_string).collect(),
                    Value::StringArray(sa) => {
                        let mut fields_string = String::new();
                        for string in sa.iter().flatten() {
                            fields_string.push_str(string);
                            fields_string.push('|');
                        }
                        fields_string
                            .splitn(split_csq, '|')
                            .map(ToString::to_string)
                            .collect()
                    }
                    _ => panic!("CSQ field not string(s)!"),
                };
                if subfields.len() < split_csq {
                    for _ in 0..(split_csq - subfields.len()) {
                        subfields.push("NA".to_owned());
                    }
                }
                fields.append(&mut subfields);
            }
            // non-CSQ field
            _ => fields.push(match info.get(info_field) {
                None => "NA".to_owned(),
                Some(field) => match field.value() {
                    Some(vcf::record::info::field::Value::Flag) => "true".to_string(),
                    Some(f) => f.to_string(),
                    None => "missing".to_string(),
                },
            }),
        }
    }
}

fn main_fields_from_record(record: &vcf::Record) -> Vec<String> {
    vec![
        contig_string(record.chromosome()),          // chr
        format!("{}", i32::from(record.position())), // pos
        record // id
            .ids()
            .iter()
            .map(ToString::to_string)
            .collect::<Vec<String>>()
            .join(","),
        format!("{}", record.reference_bases()), // ref
        format!("{}", record.alternate_bases()), // alt
        // qual
        if let Some(qual) = record.quality_score() {
            format!("{}", qual)
        } else {
            "NA".to_owned()
        },
        // filter
        if let Some(filters) = record.filters() {
            match filters {
                Filters::Pass => "PASS".to_string(),
                Filters::Fail(reasons) => reasons
                    .iter()
                    .map(ToString::to_string)
                    .collect::<Vec<String>>()
                    .join(","),
            }
        } else {
            "NA".to_owned()
        },
    ]
}

fn contig_string(contig: &Chromosome) -> String {
    match contig {
        Chromosome::Name(s) => s.to_owned(),
        Chromosome::Symbol(s) => s.to_owned(),
    }
}
