use noodles_vcf as vcf;
use structopt::StructOpt;
use vcf::record::chromosome::Chromosome;
use vcf::record::filters::Filters;

use std::error::Error;
use std::fs::File;
use std::io::{stdin, stdout, BufReader, Read, Write};

mod cli;

use cli::Opt;

const HEADER_START: [&str; 7] = ["contig", "pos", "id", "ref", "alt", "qual", "filter"];

enum TableKind {
    Cartesian,
    Stacked,
}

fn main() -> Result<(), Box<dyn Error>> {
    let opt = Opt::from_args();

    let kind = match opt.stack {
        true => TableKind::Stacked,
        false => TableKind::Cartesian,
    };

    let input: Box<dyn Read> = match opt.input.clone() {
        Some(path) => Box::new(File::open(path)?),
        None => Box::new(stdin()),
    };

    let output: Box<dyn Write> = match opt.output.clone() {
        Some(path) => Box::new(File::create(path)?),
        None => Box::new(stdout()),
    };

    convert_records(input, output, kind, opt)?;

    Ok(())
}

fn convert_records(
    input: Box<dyn Read>,
    writer: Box<dyn Write>,
    kind: TableKind,
    opt: Opt,
) -> Result<(), Box<dyn Error>> {
    let mut reader = vcf::Reader::new(BufReader::new(input));

    let writer = match opt.csv {
        true => csv::WriterBuilder::new()
            .delimiter(b',')
            .from_writer(writer),
        false => csv::WriterBuilder::new()
            .delimiter(b'\t')
            .from_writer(writer),
    };

    let header: vcf::Header = reader.read_header()?.parse()?;

    let info_prefix = match opt.info_prefix {
        Some(p) => p,
        None => "info_".to_owned(),
    };

    match kind {
        TableKind::Cartesian => {
            let sample_delim = match opt.sample_delim {
                Some(d) => d,
                None => "_".to_string(),
            };
            convert_records_cartesian(reader, writer, header, sample_delim, info_prefix)?
        }
        TableKind::Stacked => convert_records_stacked(reader, writer, header, info_prefix)?,
    }
    Ok(())
}

fn convert_records_stacked(
    mut reader: vcf::Reader<BufReader<impl Read>>,
    mut writer: csv::Writer<impl Write>,
    header: vcf::Header,
    info_prefix: String,
) -> Result<(), Box<dyn Error>> {
    let fmts = header.formats();
    let infos = header.infos();
    let samples = header.sample_names();

    let mut info_header: Vec<String> = infos
        .keys()
        .map(|field| format!("{}{}", info_prefix, field))
        .collect();

    let mut out_header: Vec<String> = HEADER_START.map(ToString::to_string).to_vec();
    out_header.append(&mut info_header);

    let mut fmt_header: Vec<String> = fmts
        .keys()
        .map(ToString::to_string)
        .collect::<Vec<String>>();

    fmt_header.push("sample".to_owned());

    out_header.append(&mut fmt_header);
    writer.write_record(&out_header)?;

    for result in reader.records() {
        let record = result?;

        let mut locus_fields = main_fields_from_record(&record);

        push_info_fields(infos, record.info(), &mut locus_fields);

        let sample_fmts = record.genotypes();

        for (sample_fmt, sample) in sample_fmts.iter().zip(samples) {
            let mut fields = locus_fields.clone();
            for key in fmts.keys() {
                if let Some(field) = sample_fmt.get(key) {
                    fields.push(field.to_string());
                } else {
                    fields.push("NA".to_owned());
                }
            }
            fields.push(sample.clone());
            writer.write_record(&fields)?;
        }
    }
    Ok(())
}

fn convert_records_cartesian(
    mut reader: vcf::Reader<BufReader<impl Read>>,
    mut writer: csv::Writer<impl Write>,
    header: vcf::Header,
    sample_delim: String,
    info_prefix: String,
) -> Result<(), Box<dyn Error>> {
    let fmts = header.formats();
    let infos = header.infos();
    let samples = header.sample_names();

    let mut info_header: Vec<String> = infos
        .keys()
        .map(|field| format!("{}{}", info_prefix, field))
        .collect();

    let mut out_header: Vec<String> = HEADER_START.map(ToString::to_string).to_vec();
    out_header.append(&mut info_header);
    let mut fmt_header: Vec<String> = samples
        .iter()
        .map(|sample| {
            fmts.keys()
                .map(|field| format!("{}{}{}", sample, sample_delim, field))
                .collect::<Vec<String>>()
        })
        .flatten()
        .collect();

    out_header.append(&mut fmt_header);
    writer.write_record(&out_header)?;

    for result in reader.records() {
        let record = result?;

        let mut fields = main_fields_from_record(&record);

        push_info_fields(infos, record.info(), &mut fields);

        let sample_fmts = record.genotypes();

        for sample_fmt in sample_fmts.iter() {
            for key in fmts.keys() {
                if let Some(field) = sample_fmt.get(key) {
                    fields.push(field.to_string());
                } else {
                    fields.push("NA".to_owned());
                }
            }
        }

        writer.write_record(&fields)?;
    }
    Ok(())
}

fn push_info_fields(
    infos: &vcf::header::Infos,
    info: &vcf::record::Info,
    fields: &mut Vec<String>,
) {
    for info_field in infos.keys() {
        fields.push(match info.get(info_field) {
            Some(field) => match field.value() {
                vcf::record::info::field::Value::Flag => "true".to_string(),
                _ => field.value().to_string(),
            },
            None => "NA".to_owned(),
        })
    }
}

fn main_fields_from_record(record: &vcf::Record) -> Vec<String> {
    vec![
        contig_string(record.chromosome()),          // chr
        format!("{}", i32::from(record.position())), // pos
        record // id
            .ids()
            .iter()
            .map(ToString::to_string)
            .collect::<Vec<String>>()
            .join(","),
        format!("{}", record.reference_bases()), // ref
        format!("{}", record.alternate_bases()), // alt
        // qual
        if let Some(qual) = record.quality_score() {
            format!("{}", qual)
        } else {
            "NA".to_owned()
        },
        // filter
        if let Some(filters) = record.filters() {
            match filters {
                Filters::Pass => "PASS".to_string(),
                Filters::Fail(reasons) => reasons
                    .iter()
                    .map(ToString::to_string)
                    .collect::<Vec<String>>()
                    .join(","),
            }
        } else {
            "NA".to_owned()
        },
    ]
}

fn contig_string(contig: &Chromosome) -> String {
    match contig {
        Chromosome::Name(s) => s.to_owned(),
        Chromosome::Symbol(s) => s.to_owned(),
    }
}
