use std::sync::Arc;

use crate::isahc::http_client;
use crate::redirect_strategy::{HttpRedirect, RedirectStrategy};
use crate::request_ext::OpenIdConnectRequestExtData;
use openidconnect::{
    core::{CoreClient, CoreProviderMetadata, CoreResponseType},
    AccessToken, AuthenticationFlow, AuthorizationCode, ClientId, ClientSecret, CsrfToken,
    IssuerUrl, Nonce, OAuth2TokenResponse, RedirectUrl, Scope, SubjectIdentifier,
};
use serde::{Deserialize, Serialize};
use tide::{http::Method, Middleware, Next, Redirect, Request, StatusCode};

const SESSION_KEY: &str = "tide.oidc";

/// Middleware configuration.
#[derive(Debug, Deserialize)]
pub struct Config {
    /// Issuer URL used to A) retrieve the OpenID Connect provider
    /// metadata *and* B) validate that tokens are generated by the
    /// expected issuer.
    pub issuer_url: IssuerUrl,

    /// Our Client ID, as generated by the OpenID Connect provider.
    pub client_id: ClientId,

    /// Our Client Secret, as generated by the OpenID Connect provider.
    pub client_secret: ClientSecret,

    /// URL to which the OpenID Connect provider will redirect authenticated
    /// requests; must be a URL registered with the provider.
    pub redirect_url: RedirectUrl,

    /// Optional URL used to log the user out of the identity provider
    /// as part of the application logout process. If provided, the
    /// browser will be redirected to this URL *after* clearing the auth
    /// info from the Tide session.
    ///
    /// Note that most Identity Providers allow you to include a
    /// provider-specific query parameter in this URL. The browser will
    /// be redirected to that URL after the logout process has been
    /// completed and the provider's credentials have been cleared from
    /// the browser.
    ///
    /// You should almost certainly populate that query parameter so
    /// that the user ends up at your site after the logout. In most
    /// cases the value of this parameter will be the full URL to your
    /// site's
    /// [`logout_landing_path`](OpenIdConnectMiddleware::with_logout_landing_path).
    ///
    /// Finally, identity providers often require you to register the
    /// logout URL in their configuration, usually in the same place where
    /// you register your [redirect URL](Self::redirect_url).
    pub idp_logout_url: Option<String>,
}

#[derive(Debug, Deserialize, Serialize)]
enum MiddlewareSessionState {
    PreAuth(CsrfToken, Nonce),
    PostAuth(SubjectIdentifier, AccessToken, Vec<Scope>),
}

/// Open ID Connect Middleware.
pub struct OpenIdConnectMiddleware {
    login_path: String,
    redirect_url: RedirectUrl,
    scopes: Vec<Scope>,
    login_landing_path: String,
    logout_path: String,
    logout_destroys_session: bool,
    idp_logout_url: Option<String>,
    logout_landing_path: String,
    client: CoreClient,
    redirect_strategy: Arc<dyn RedirectStrategy>,
}

impl std::fmt::Debug for OpenIdConnectMiddleware {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.debug_struct("OpenIdConnectMiddleware")
            .field("login_path", &self.login_path)
            .field("scopes", &self.scopes)
            .field("redirect_url", &self.redirect_url)
            .field("login_landing_path", &self.login_landing_path)
            .field("idp_logout_url", &self.idp_logout_url)
            .field("logout_path", &self.logout_path)
            .field("logout_destroys_session", &self.logout_destroys_session)
            .field("logout_landing_path", &self.logout_landing_path)
            .finish()
    }
}

impl OpenIdConnectMiddleware {
    /// Create a new instance.
    ///
    /// Requests the Identity Provider's metadata and uses that to initialize
    /// various provider-specific configuration inside of the middleware.
    ///
    /// # Panics
    ///
    /// Panics if the OpenID Connect provider metadata could not be
    /// retrieved or does not match the configured
    /// [`issuer_url`](Config::issuer_url).
    ///
    /// # Defaults
    ///
    /// The defaults for OpenIdConnectMiddleware are:
    /// - redirect strategy: [`HttpRedirect`](crate::redirect_strategy::HttpRedirect)
    /// - login path: `/login`
    /// - scopes: `["openid"]`
    /// - login landing path: `/`
    /// - logout path: `/logout`
    /// - logout destroys session: `true`
    /// - logout landing path: `/`
    ///
    /// # Examples
    ///
    /// ```no_run
    /// use tide_openidconnect::{self};
    ///
    /// # async_std::task::block_on(async {
    /// let config = tide_openidconnect::Config {
    ///     // ... set/load config ...
    /// #   issuer_url: tide_openidconnect::IssuerUrl::new("https://your-tenant-name.us.auth0.com/".to_string()).unwrap(),
    /// #   client_id: tide_openidconnect::ClientId::new("app-id-goes-here".to_string()),
    /// #   client_secret: tide_openidconnect::ClientSecret::new("app-secret-goes-here".to_string()),
    /// #   redirect_url: tide_openidconnect::RedirectUrl::new("http://your.cool.site/callback".to_string()).unwrap(),
    /// #   idp_logout_url: None,
    /// };
    /// let middleware = tide_openidconnect::OpenIdConnectMiddleware::new(&config)
    ///     .await
    ///     .with_logout_landing_path("/loggedout");
    /// # })
    /// ```
    pub async fn new(config: &Config) -> Self {
        // Get the OpenID Connect provider metadata.
        let provider_metadata =
            CoreProviderMetadata::discover_async(config.issuer_url.clone(), http_client)
                .await
                .expect("Unable to load OpenID Connect provider metadata.");

        // Create the OpenID Connect client.
        let client = CoreClient::from_provider_metadata(
            provider_metadata,
            config.client_id.clone(),
            Some(config.client_secret.clone()),
        )
        .set_redirect_uri(config.redirect_url.clone());

        // Initialize the middleware with our defaults. Note that we do not
        // have to include "openid" in the (default) scopes, because the
        // openidconnect-rs crate always adds that to the scopes list.
        let login_path = "/login".to_string();
        Self {
            login_path: login_path.clone(),
            scopes: vec![],
            redirect_url: config.redirect_url.clone(),
            login_landing_path: "/".to_string(),
            client,
            redirect_strategy: Arc::new(HttpRedirect::new(login_path)),
            logout_path: "/logout".to_string(),
            logout_destroys_session: true,
            idp_logout_url: config.idp_logout_url.clone(),
            logout_landing_path: "/".to_string(),
        }
    }

    /// Sets the path to the "login" route that will be intercepted by the
    /// middleware in order to redirect the browser to the OpenID Connect
    /// authentication page.
    ///
    /// Defaults to `/login`
    pub fn with_login_path(mut self, login_path: &str) -> Self {
        self.login_path = login_path.to_string();
        self
    }

    /// Adds one or more scopes to the OpenID Connect request.
    ///
    /// Defaults to `openid` (which is the minimum required scope).
    pub fn with_scopes(mut self, scopes: &[impl AsRef<str>]) -> Self {
        self.scopes = scopes
            .iter()
            .map(|s| Scope::new(s.as_ref().to_owned()))
            .collect();
        self
    }

    /// Sets the path where the browser will be sent after a successful
    /// login sequence.
    ///
    /// Defaults to `/`
    pub fn with_login_landing_path(mut self, login_landing_path: &str) -> Self {
        self.login_landing_path = login_landing_path.to_string();
        self
    }

    /// Sets the path to the "logout" route that will be intercepted by
    /// the middleware in order to clear the sessions's authentication
    /// state.
    ///
    /// Defaults to `/logout`
    pub fn with_logout_path(mut self, logout_path: &str) -> Self {
        self.logout_path = logout_path.to_string();
        self
    }

    /// Sets a flag indicating if the logout URL should destroy *all*
    /// session state -- both the auth state *and* any app-level state --
    /// or if logout should clear only the auth state and leave the remainder
    /// of the state intact.
    ///
    /// Applications should only retain session state after a logout if
    /// doing so will *not* leave any (private) artifacts of the user's
    /// data in the session after the logout process. The current page,
    /// status of collapsed UI elements, etc. would be safe to retain,
    /// whereas the user name, form content, etc. needs to be cleared
    /// after the logout process (which can be done by, for example,
    /// configuring the
    /// [`logout_landing_path`](Self::with_logout_landing_path) to go to
    /// a route that cleans up personally-identifying information after
    /// the logout completes).
    ///
    /// Defaults to `true`
    pub fn with_logout_destroys_session(mut self, logout_destroys_session: bool) -> Self {
        self.logout_destroys_session = logout_destroys_session;
        self
    }

    /// Sets the path where the browser will be sent after the logout
    /// sequence.
    ///
    /// Defaults to `/`
    pub fn with_logout_landing_path(mut self, logout_landing_path: &str) -> Self {
        self.logout_landing_path = logout_landing_path.to_string();
        self
    }

    /// Sets the trait used to generate redirect responses to
    /// unauthenticated requests.
    ///
    /// Defaults to [`HttpRedirect`](crate::redirect_strategy::HttpRedirect)
    pub fn with_unauthenticated_redirect_strategy<R>(mut self, redirect_strategy: R) -> Self
    where
        R: RedirectStrategy + 'static,
    {
        self.redirect_strategy = Arc::new(redirect_strategy);
        self
    }

    async fn generate_redirect<State>(&self, mut req: Request<State>) -> tide::Result
    where
        State: Clone + Send + Sync + 'static,
    {
        let mut request = self.client.authorize_url(
            AuthenticationFlow::<CoreResponseType>::AuthorizationCode,
            CsrfToken::new_random,
            Nonce::new_random,
        );
        for s in &self.scopes {
            request = request.add_scope(s.clone());
        }
        let (authorize_url, csrf_token, nonce) = request.url();

        // Initialize the middleware's session state so that we can
        // validate the login after the user completes the authentication
        // flow.
        req.session_mut()
            .insert(
                SESSION_KEY,
                MiddlewareSessionState::PreAuth(csrf_token, nonce),
            )
            .map_err(|error| tide::http::Error::new(StatusCode::InternalServerError, error))?;

        Ok(Redirect::new(&authorize_url).into())
    }

    async fn handle_callback<State>(&self, mut req: Request<State>) -> tide::Result
    where
        State: Clone + Send + Sync + 'static,
    {
        // Get the middleware state from the session. If this fails then
        // A) the browser got to the callback URL without actually going
        // through the auth process, or B) more likely, the session
        // middleware is configured with Strict cookies instead of Lax
        // cookies. We cannot tell at this level which error occurred,
        // so we just reject the request and log the error.
        if let Some(MiddlewareSessionState::PreAuth(csrf_token, nonce)) =
            req.session().get(SESSION_KEY)
        {
            // Extract the OpenID callback information and verify the CSRF
            // state.
            #[derive(Deserialize)]
            struct OpenIdCallback {
                code: AuthorizationCode,
                state: String,
            }
            let callback_data: OpenIdCallback = req.query()?;
            if &callback_data.state != csrf_token.secret() {
                return Err(tide::http::Error::from_str(
                    StatusCode::Unauthorized,
                    "Invalid CSRF state.",
                ));
            }

            // Exchange the code for a token.
            let token_response = self
                .client
                .exchange_code(callback_data.code)
                .request_async(http_client)
                .await
                .map_err(|error| tide::http::Error::new(StatusCode::InternalServerError, error))?;

            // Get the claims and verify the nonce.
            let claims = token_response
                .extra_fields()
                .id_token()
                .ok_or_else(|| {
                    tide::http::Error::from_str(
                        StatusCode::InternalServerError,
                        "OpenID Connect server did not return an ID token.",
                    )
                })?
                .claims(&self.client.id_token_verifier(), &nonce)
                .map_err(|error| tide::http::Error::new(StatusCode::Unauthorized, error))?;

            // Add the user id to the session state in order to mark this
            // session as authenticated.
            req.session_mut()
                .insert(
                    SESSION_KEY,
                    MiddlewareSessionState::PostAuth(
                        claims.subject().clone(),
                        token_response.access_token().clone(),
                        token_response.scopes().unwrap_or(&self.scopes).clone(),
                    ),
                )
                .map_err(|error| tide::http::Error::new(StatusCode::InternalServerError, error))?;

            // The user has logged in; redirect them to the main site.
            Ok(Redirect::new(&self.login_landing_path).into())
        } else {
            tide::log::warn!(
                    "Missing OpenID Connect state in session; make sure SessionMiddleware is configured with SameSite::Lax (but do *not* mutate server-side state on GET requests if you make that change!)."
                );
            Err(tide::http::Error::from_str(
                StatusCode::InternalServerError,
                "Missing authorization state.",
            ))
        }
    }
}

#[tide::utils::async_trait]
impl<State> Middleware<State> for OpenIdConnectMiddleware
where
    State: Clone + Send + Sync + 'static,
{
    async fn handle(&self, mut req: Request<State>, next: Next<'_, State>) -> tide::Result {
        // Is this URL one of the URLs that we need to intercept as part
        // of the OpenID Connect auth process? If so, apply the appropriate
        // part of the auth process according to the URL. If not, verify
        // that the request is authenticated, and if not, redirect the
        // browser to the login URL. And if they are authenticated, then
        // just proceed to the handler (after populating the request extension
        // fields).
        if req.method() == Method::Get && req.url().path() == self.login_path {
            self.generate_redirect(req).await
        } else if req.method() == Method::Get && req.url().path() == self.redirect_url.url().path()
        {
            self.handle_callback(req).await
        } else if req.method() == Method::Get && req.url().path() == self.logout_path {
            // Destroy the session as part of the logout, or clear only
            // the app state, depending on how the middleware has been
            // configured.
            if self.logout_destroys_session {
                req.session_mut().destroy();
            } else {
                req.session_mut().remove(SESSION_KEY);
            }

            // Redirect the user now that their authentication state has
            // been cleared; we send them either to the identity provider's
            // logout URL (if provided), or to the app's logout landing
            // path if the app is not configured to log the user out of
            // the identity provider.
            if let Some(idp_logout_url) = &self.idp_logout_url {
                Ok(Redirect::new(idp_logout_url).into())
            } else {
                Ok(Redirect::new(&self.logout_landing_path).into())
            }
        } else {
            // Get the middleware's session state (which will *not* be
            // present if the browser has not yet gone through the auth
            // process), then augment the request with the authentication
            // status.
            match req.session().get(SESSION_KEY) {
                Some(MiddlewareSessionState::PostAuth(subject, access_token, scopes)) => req
                    .set_ext(OpenIdConnectRequestExtData::Authenticated {
                        user_id: subject.to_string(),
                        access_token: access_token.secret().to_string(),
                        scopes: scopes.iter().map(|s| s.to_string()).collect(),
                    }),
                _ => req.set_ext(OpenIdConnectRequestExtData::Unauthenticated {
                    redirect_strategy: self.redirect_strategy.clone(),
                }),
            };

            // Call the downstream middleware.
            Ok(next.run(req).await)
        }
    }
}
