use async_trait::async_trait;
use tide::{Middleware, Next, Request};

mod basic;
mod bearer;
mod error;
mod traits;

pub use basic::Basic;
pub use bearer::Bearer;
pub use error::{Error, Result};
pub use traits::{AuthValue, Scheme};

pub struct Auth<S: Scheme + Send + Sync>(S);

#[async_trait]
impl<S: Scheme + Send + Sync> Scheme for Auth<S> {
    fn header_name() -> http_types::headers::HeaderName {
        S::header_name()
    }

    async fn parse(&self, header_value: &http_types::headers::HeaderValue) -> Result<AuthValue> {
        self.0.parse(header_value).await
    }
}

impl<S: Scheme + Send + Sync> From<S> for Auth<S> {
    fn from(scheme: S) -> Self {
        Auth(scheme)
    }
}

#[async_trait]
impl<State, S> Middleware<State> for Auth<S>
where
    State: Clone + Send + Sync + 'static,
    S: Scheme + Send + Sync + 'static,
{
    async fn handle(&self, mut request: Request<State>, next: Next<'_, State>) -> tide::Result {
        let res = if let None = request.ext::<AuthValue>() {
            let header_value = match request.header(&S::header_name()) {
                Some(header_value) => header_value,
                _ => return Err(Error::Invalid.into()),
            };

            let value = self.parse(header_value).await?;

            request.set_ext(value);

            next.run(request).await
        } else {
            next.run(request).await
        };

        Ok(res)
    }
}
